package ru.yandex.chemodan.app.dataapi.worker.dump.full;

import java.util.List;
import java.util.stream.Collectors;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;

import static ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils.YT_NODE_NAME_FORMATTER;

/**
 * @author metal
 */
public class YtDumpBackupManager {
    private final Yt yt;
    private final RetryPolicy ytRetryPolicy;
    private final int numberOfDaysToBackup;

    public YtDumpBackupManager(Yt yt, RetryPolicy ytRetryPolicy, int numberOfDaysToBackup) {
        this.numberOfDaysToBackup = numberOfDaysToBackup;
        this.yt = yt;
        this.ytRetryPolicy = ytRetryPolicy;
    }

    public void manageBackupsForPath(YPath path) {
        makeBackup(path);
        cleanOldBackups(path.parent());
    }

    private void cleanOldBackups(YPath folderPath) {
        List<YPath> backups = yt.cypress()
                .list(folderPath)
                .stream()
                .map(YTreeStringNode::getValue)
                .filter(date -> !date.equals(YtPathsUtils.CURRENT_DUMP_NAME))
                .filter(YtDumpBackupManager::hasProperTableNameFormat)
                .sorted()
                .map(folderPath::child)
                .collect(Collectors.toList());
        for (int i = 0; i < backups.size() - numberOfDaysToBackup; i++) {
            yt.cypress().remove(backups.get(i));
        }
    }

    public static boolean hasProperTableNameFormat(String name) {
        return getDateTimeFromYtNodeName(name).isPresent();
    }

    public static Option<DateTime> getDateTimeFromYtNodeName(String name) {
        try {
            return Option.of(YtPathsUtils.YT_NODE_NAME_FORMATTER.parseDateTime(name));
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }

    private void makeBackup(YPath path) {
        DateTime now = DateTime.now();
        String yesterday = YT_NODE_NAME_FORMATTER.print(now.minusDays(1));
        YPath yesterdayBackup = path.parent().child(yesterday);
        move(path, yesterdayBackup);
    }

    private void move(YPath source, YPath destination) {
        new RetryManager()
                .withRetryPolicy(ytRetryPolicy)
                .withLogging("Moving backups: " + source + " -> " + destination)
                .runSafe(() -> {
                    if (yt.cypress().exists(source)) {
                        yt.cypress().copy(source, destination, true, true, true);
                    }
                });
    }
}
