package ru.yandex.chemodan.app.dataapi.worker.dump.full;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class YtDumpConsistencyManager {
    private static final Logger logger = LoggerFactory.getLogger(YtDumpConsistencyManager.class);

    private final YtDumpAndChangesMerger ytDumpAndChangesMerger;
    private final YtDumpAndChangesMergeScriptManager ytDumpAndChangesMergeScriptManager;
    private final CollectionNamesExtractor collectionNamesExtractor;
    private final YtDumpBackupManager ytDumpBackupManager;

    private final AppSettingsRegistry appSettingsRegistry;
    private final ExecutorService executorService;

    private final DynamicProperty<Boolean> useParallelDumper = DynamicProperty.cons("yt-dump-parallel-version-enabled", true);

    public YtDumpConsistencyManager(YtDumpAndChangesMerger ytDumpAndChangesMerger,
            YtDumpAndChangesMergeScriptManager ytDumpAndChangesMergeScriptManager,
            CollectionNamesExtractor collectionNamesExtractor, YtDumpBackupManager ytDumpBackupManager,
            AppSettingsRegistry appSettingsRegistry,
            ExecutorService executorService)
    {
        this.ytDumpAndChangesMerger = ytDumpAndChangesMerger;
        this.ytDumpAndChangesMergeScriptManager = ytDumpAndChangesMergeScriptManager;
        this.collectionNamesExtractor = collectionNamesExtractor;
        this.ytDumpBackupManager = ytDumpBackupManager;
        this.appSettingsRegistry = appSettingsRegistry;
        this.executorService = executorService;
    }

    public void updateDatabases() {
        ytDumpAndChangesMergeScriptManager.uploadMergeScriptInYt();

        if (useParallelDumper.get()) {
            ListF<CompletableFuture<Object>> futures = getDatabasesToUpdate().flatMap(this::updateDatabaseParallel);

            try {
                CompletableFutures.allOf(futures).get();
            } catch (Exception e) {
                logger.error("Failed to dump collections", e);
            }
        } else {
            getDatabasesToUpdate().forEach(this::updateDatabase);
        }
    }

    public void updateDatabase(DatabaseRef databaseRef) {
        collectionNamesExtractor
                .getCollectionNames(databaseRef)
                .map(databaseRef::consColRef)
                .forEach(this::updateSpecificCollectionSafe);
    }

    public ListF<CompletableFuture<Object>> updateDatabaseParallel(DatabaseRef databaseRef) {
        return collectionNamesExtractor
                .getCollectionNames(databaseRef)
                .map(databaseRef::consColRef)
                .map(collectionRef -> CompletableFuture.supplyAsync(
                        YandexCloudRequestIdHolder.supplyWithYcrid(
                                () -> updateSpecificCollectionSafe(collectionRef)
                        ), executorService));
    }

    public Void updateSpecificCollectionSafe(CollectionRef collectionRef) {
        try {
            updateSpecificCollection(collectionRef);
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            logger.warn("Failed to update collection {}: {}", collectionRef, t);
        }
        return null;
    }

    public void updateSpecificCollection(CollectionRef collectionRef) {
        logger.info("Updating in yt dump of " + collectionRef);

        YPath dumpFolder = YtPathsUtils.getProperYPath(
                FullDumpDatabaseUsersProcessor.DATASYNC_FULL_DUMP_YT_PATH_PREFIX,
                collectionRef);
        YPath dumpPath = YtPathsUtils.getProperYPathForCurrentDump(dumpFolder);
        ytDumpBackupManager.manageBackupsForPath(dumpPath);

        ytDumpAndChangesMerger.mergeForDatabase(collectionRef);

        logger.info("Dump updating finished for " + collectionRef);
    }

    private ListF<DatabaseRef> getDatabasesToUpdate() {
        return appSettingsRegistry
                .getAll()
                .filter(AppDatabaseSettings::isHaveDumpInYt)
                .map(settings -> DatabaseRef.cons(settings.appName, settings.databaseId.get()));
    }
}
