package ru.yandex.chemodan.app.dataapi.worker.dump.full;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeIntegerNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeMapNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils.CURRENT_DUMP_NAME;
import static ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils.YT_NODE_NAME_FORMATTER;

/**
 * @author metal
 */
public class YtDumpConsistencyManagerTest {
    private static final String PATH = "//home/datasync-api/app/db/col";
    private static final String CURRENT_PATH = PATH + "/" + CURRENT_DUMP_NAME;
    private static final int NUMBER_OF_DAYS_TO_BACKUP = 5;

    private Yt yt;
    private Cypress cypress;

    @Before
    public void init() {
        cypress = Mockito.mock(Cypress.class);
        Mockito
                .when(cypress.exists(any(YPath.class)))
                .thenReturn(true);

        yt = Mockito.mock(Yt.class);
        Mockito
                .when(yt.cypress())
                .thenReturn(cypress);

        Mockito
                .when(cypress.list(YPath.simple(PATH)))
                .thenReturn(Cf.list(
                        YTree.stringNode(CURRENT_DUMP_NAME),
                        YTree.stringNode("2016-07-20"),
                        YTree.stringNode("2016-06-30"),
                        YTree.stringNode("2016-07-10"),
                        YTree.stringNode("2016-06-20"),
                        YTree.stringNode("2016-07-30"),
                        YTree.stringNode(YT_NODE_NAME_FORMATTER.print(DateTime.now().minusDays(1))),
                        YTree.stringNode("2016-05-25"),
                        YTree.stringNode("2016-08-12"),
                        YTree.stringNode("wrong_format")));
    }

    @Test
    public void dumpBackupTest() {
        YtDumpBackupManager manager = new YtDumpBackupManager(yt, new RetryPolicy(), NUMBER_OF_DAYS_TO_BACKUP);
        manager.manageBackupsForPath(YPath.simple(CURRENT_PATH));

        verify(cypress, times(1)).exists(any(YPath.class));

        ArgumentCaptor<YPath> sourceYPathCaptor = ArgumentCaptor.forClass(YPath.class);
        ArgumentCaptor<YPath> destinationYPathCaptor = ArgumentCaptor.forClass(YPath.class);
        verify(cypress, times(1))
                .copy(sourceYPathCaptor.capture(), destinationYPathCaptor.capture(), eq(true), eq(true), eq(true));

        ListF<YPath> capturedSources = Cf.x(sourceYPathCaptor.getAllValues());
        ListF<YPath> destinationSources = Cf.x(destinationYPathCaptor.getAllValues());

        DateTime now = DateTime.now();
        Assert.equals(CURRENT_PATH, capturedSources.get(0).toString());
        Assert.equals(PATH + "/" + YT_NODE_NAME_FORMATTER.print(now.minusDays(1)), destinationSources.get(0).toString());

        ArgumentCaptor<YPath> removeYPathCaptor = ArgumentCaptor.forClass(YPath.class);
        verify(cypress, times(3)).remove(removeYPathCaptor.capture());
        ListF<YPath> removeSources = Cf.x(removeYPathCaptor.getAllValues());
        Assert.equals(PATH + "/2016-05-25", removeSources.get(0).toString());
        Assert.equals(PATH + "/2016-06-20", removeSources.get(1).toString());
        Assert.equals(PATH + "/2016-06-30", removeSources.get(2).toString());
    }

    private void setupRowCountAnswer(YPath path, long rowCount) {
        Mockito.when(cypress.get(path.attribute("row_count")))
                .thenReturn(new YTreeIntegerNodeImpl(false, rowCount, Cf.map()));
    }

    private void setupGetNodeAnswer(YPath path, ListF<String> nodes) {
        YTreeNode getResult = new YTreeMapNodeImpl(Cf.map());
        Cf.wrap(getResult.asMap()).putAll(nodes.zipWith(YTree::stringNode));
        Mockito.when(cypress.get(path)).thenReturn(getResult);
    }

    @Test
    public void getChangesPathsInYtTest() {
        YPath dumpPath = YPath.simple("//home/dump");
        YPath logPath = YPath.simple("//home/log");
        YPath currentTable = dumpPath.child(CURRENT_DUMP_NAME);
        YPath justCreatedTable = dumpPath.child("2018-07-07");
        YPath recentEmptyTable = dumpPath.child("2018-07-04");
        YPath recentNotEmptyTable = dumpPath.child("2018-07-03");
        YPath ancientNotEmptyTable = dumpPath.child("2018-06-01");

        ListF<String> dumpNodes =
                Cf.list(currentTable, justCreatedTable, recentEmptyTable, recentNotEmptyTable, ancientNotEmptyTable)
                        .map(YPath::name);

        ListF<String> logNodes = Cf.range(1, 9).map(day -> "2018-07-0" + day);

        setupGetNodeAnswer(dumpPath, dumpNodes);

        setupRowCountAnswer(recentEmptyTable, 0);
        setupRowCountAnswer(recentNotEmptyTable, 1);
        setupRowCountAnswer(ancientNotEmptyTable, 1);

        setupGetNodeAnswer(logPath, logNodes);

        YtDumpAndChangesMerger merger = new YtDumpAndChangesMerger(yt, new RetryPolicy());
        ListF<String> logs = merger.getChangesPathsInYtForTest(dumpPath, logPath).map(YPath::name);

        Assert.hasSize(6, logs);
        Assert.equals("2018-07-03", logs.min());
    }
}
