package ru.yandex.chemodan.app.dataapi.worker.importer;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils;
import ru.yandex.chemodan.app.dataapi.worker.importer.processors.SingleDiffProcessorFactory;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ImportDataReader;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic.GenericObjectsProcessor;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic.ProcessingContext;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.net.HostnameUtils;

/**
 * @author metal
 */
public class ImportDataChunkTask extends OnetimeTaskSupport<ImportDataChunkTask.Parameters> {
    public static final TaskQueueName IMPORT_DATA_QUEUE = new TaskQueueName("dataapi-import-data-from-yt");

    private ImportDataReader importDataReader;
    private SingleDiffProcessorFactory singleDiffProcessorFactory;

    private TypeSettingsRegistry typeSettingsRegistry;
    private GenericObjectsProcessor genericObjectsProcessor;

    private ImportDataChunkTask(Parameters parameters) {
        super(parameters);
    }

    public static ImportDataChunkTask staticTyped(
            long lowerIndex, long upperIndex, String path, CollectionRef collectionRef)
    {
        return new ImportDataChunkTask(new Parameters(lowerIndex, upperIndex, path,
                new StaticTypeParameters(collectionRef), new GenericTypeParameters()));
    }

    public static ImportDataChunkTask genericTyped(
            long lowerIndex, long upperIndex, String path, String typeName, String taskId)
    {
        return new ImportDataChunkTask(new Parameters(lowerIndex, upperIndex, path,
                new StaticTypeParameters(), new GenericTypeParameters(typeName, taskId)));
    }

    public ImportDataChunkTask(
            ImportDataReader importDataReader,
            SingleDiffProcessorFactory singleDiffProcessorFactory,
            TypeSettingsRegistry typeSettingsRegistry,
            GenericObjectsProcessor genericObjectsProcessor)
    {
        super(Parameters.class);
        this.importDataReader = importDataReader;
        this.singleDiffProcessorFactory = singleDiffProcessorFactory;
        this.typeSettingsRegistry = typeSettingsRegistry;
        this.genericObjectsProcessor = genericObjectsProcessor;
    }

    @Override
    public TaskQueueName queueName() {
        return IMPORT_DATA_QUEUE;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(15);
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) throws Exception {
        YPath pathWithRange = YPath.simple(parameters.path).withRange(parameters.lowerRowIndex, parameters.upperRowIndex);

        if (parameters.staticType.app.isPresent()) {
            CollectionRef collection = parameters.staticType.getCollectionRef();
            Function2V<DataApiUserId, Delta> callback = singleDiffProcessorFactory.processSingleDiffF(
                    parameters.staticType.app.get(), parameters.staticType.dbId.get());

            importDataReader.forEachDiff(pathWithRange, collection, callback);

        } else if (parameters.genericType.typeName.isPresent()) {
            TypeSettings type = typeSettingsRegistry.getTypeSettings(parameters.genericType.typeName.get());

            ProcessingContext ctx = new ProcessingContext(
                    HostnameUtils.localHostnameShort(), context.getFullJobId().getJobId().toString());

            YPath resultPath = YtPathsUtils.getImportsResultsYPath(
                    parameters.genericType.typeName.get(), parameters.genericType.importId.get());

            genericObjectsProcessor.processAndWriteResult(type, pathWithRange, ctx, resultPath.append(true));

        } else {
            throw new IllegalStateException("Unexpected parameters");
        }
    }

    @BenderBindAllFields
    public static class Parameters {
        private final long lowerRowIndex;
        private final long upperRowIndex;
        private final String path;

        @BenderFlatten
        private final StaticTypeParameters staticType;
        @BenderFlatten
        private final GenericTypeParameters genericType;

        public Parameters(
                long lowerRowIndex, long upperRowIndex, String path,
                StaticTypeParameters staticType,
                GenericTypeParameters genericType)
        {
            this.lowerRowIndex = lowerRowIndex;
            this.upperRowIndex = upperRowIndex;
            this.path = path;
            this.staticType = staticType;
            this.genericType = genericType;
        }
    }

    @BenderBindAllFields
    public static class StaticTypeParameters {
        private final Option<String> app;
        private final Option<String> dbId;
        private final Option<String> collectionId;

        public StaticTypeParameters(CollectionRef collection) {
            this.app = Option.of(collection.dbAppId());
            this.dbId = Option.of(collection.databaseId());
            this.collectionId = Option.of(collection.collectionId());
        }

        public StaticTypeParameters() {
            this.app = Option.empty();
            this.dbId = Option.empty();
            this.collectionId = Option.empty();
        }

        public CollectionRef getCollectionRef() {
            return CollectionRef.cons(new AppDatabaseRef(app.get(), dbId.get()), collectionId.get());
        }
    }

    @BenderBindAllFields
    public static class GenericTypeParameters {
        private final Option<String> typeName;
        private final Option<String> importId;

        public GenericTypeParameters(String typeName, String importId) {
            this.typeName = Option.of(typeName);
            this.importId = Option.of(importId);
        }

        public GenericTypeParameters() {
            this.typeName = Option.empty();
            this.importId = Option.empty();
        }
    }
}
