package ru.yandex.chemodan.app.dataapi.worker.importer;

import com.fasterxml.jackson.databind.JsonNode;
import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Period;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.app.dataapi.core.generic.GenericObjectsContextConfiguration;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils;
import ru.yandex.chemodan.app.dataapi.worker.importer.diff.CalculateAddressesDiffRunner;
import ru.yandex.chemodan.app.dataapi.worker.importer.diff.CalculateAddressesDiffTask;
import ru.yandex.chemodan.app.dataapi.worker.importer.processors.BaseSingleDiffProcessorFactory;
import ru.yandex.chemodan.app.dataapi.worker.importer.processors.SingleDiffProcessorFactory;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.DataRowParser;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.DeltaConstructor;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ImportDataReader;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.address.AddressChange;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.address.AddressChangeJsonNodeParser;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.address.AddressChangesProcessor;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.address.AddressChangesYtReader;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.address.AddressDeltaConstructor;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic.GenericObjectsProcessor;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ypath.ImportDataYPathManager;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ypath.ImportYPathOverridesRegistry;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ypath.LastSuccessfulImportDateRegistry;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.BazingaWorkerApp;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.inside.yt.kosher.Yt;

/**
 * @author metal
 */
@Configuration
@Import({
        GenericObjectsContextConfiguration.class,
})
public class ImportDataConfiguration {
    @Bean
    public ImportDataCronTasksRegistry importDataCronTasksRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager,
            BazingaWorkerApp bazingaWorkerApp,
            AddressChangesProcessor addressChangesProcessor)
    {
        ImportDataCronTasksRegistry importDataCronTasksRegistry = new ImportDataCronTasksRegistry(
                zkRoot.child("datasync_import_data_cron_tasks"),
                bazingaWorkerApp.getWorkerTaskRegistry(),
                addressChangesProcessor);
        zkManager.addClient(importDataCronTasksRegistry);
        return importDataCronTasksRegistry;
    }

    @Bean
    public ImportYPathOverridesRegistry importYPathOverridesRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        ImportYPathOverridesRegistry importYPathOverridesRegistry = new ImportYPathOverridesRegistry(
                zkRoot.child("datasync_import_data_ypath_overrides"));
        zkManager.addClient(importYPathOverridesRegistry);
        return importYPathOverridesRegistry;
    }

    @Bean
    public LastSuccessfulImportDateRegistry lastSuccessfulImportDateRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        LastSuccessfulImportDateRegistry lastSuccessfulImportDateRegistry = new LastSuccessfulImportDateRegistry(
                zkRoot.child("datasync_import_data_last_success"));
        zkManager.addClient(lastSuccessfulImportDateRegistry);
        return lastSuccessfulImportDateRegistry;
    }

    @Bean
    public ImportDataYPathManager importDataYPathManager(
            LastSuccessfulImportDateRegistry lastSuccessfulImportDateRegistry,
            ImportYPathOverridesRegistry importYPathOverridesRegistry,
            Yt yt)
    {
        return new ImportDataYPathManager(lastSuccessfulImportDateRegistry, importYPathOverridesRegistry, yt);
    }

    @Bean
    public AddressChangesYtReader addressChangesYtReader(
            Yt yt,
            @Qualifier("ytOperationRetryPolicy")
            RetryPolicy ytRetryPolicy,
            DataRowParser<JsonNode, AddressChange> dataRowParser,
            DeltaConstructor<AddressChange> deltaConstructor)
    {
        return new AddressChangesYtReader(yt, ytRetryPolicy, dataRowParser, deltaConstructor);
    }

    @Bean
    public AddressChangeJsonNodeParser addressChangeJsonNodeParser() {
        return new AddressChangeJsonNodeParser();
    }

    @Bean
    public AddressDeltaConstructor addressDeltaConstructor(
            DataApiManager dataApiManager,
            @Qualifier("databaseOperationRetryPolicy")
            RetryPolicy dbRetryPolicy)
    {
        return new AddressDeltaConstructor(dataApiManager, dbRetryPolicy);
    }

    @Bean
    public AddressChangesProcessor addressChangesProcessor(
            BazingaTaskManager bazingaTaskManager,
            ImportDataYPathManager importDataYPathManager,
            @Value("${dataapi.import.data.yt.row.fetch.count}")
            int importDataRowFetchCount,
            Yt yt)
    {
        return new AddressChangesProcessor(bazingaTaskManager, importDataYPathManager, importDataRowFetchCount, yt);
    }

    @Bean
    public SingleDiffProcessorFactory singleDiffProcessorFactory(
            DataApiManager dataApiManager,
            @Qualifier("databaseOperationRetryPolicy")
            RetryPolicy dbRetryPolicy)
    {
        return new BaseSingleDiffProcessorFactory(dataApiManager, dbRetryPolicy);
    }

    @Bean
    public GenericObjectsProcessor genericObjectsProcessor(
            DataApiManager dataApiManager,
            @Qualifier("databaseOperationRetryPolicy") RetryPolicy dbRetryPolicy,
            Yt yt, @Qualifier("ytOperationRetryPolicy") RetryPolicy ytRetryPolicy)
    {
        return new GenericObjectsProcessor(dataApiManager, dbRetryPolicy, yt, ytRetryPolicy);
    }

    @Bean
    public ImportDataChunkTask importDataChunkTask(
            ImportDataReader importDataReader,
            SingleDiffProcessorFactory singleDiffProcessorFactory,
            TypeSettingsRegistry typeSettingsRegistry,
            GenericObjectsProcessor genericObjectsProcessor)
    {
        return new ImportDataChunkTask(
                importDataReader, singleDiffProcessorFactory, typeSettingsRegistry, genericObjectsProcessor);
    }

    @Bean
    public ImportGenericObjectsTask importGenericObjectsTask(
            BazingaTaskManager bazingaTaskManager, Yt yt,
            @Value("${dataapi.import.data.yt.row.fetch.count}") int importDataRowFetchCount)
    {
        return new ImportGenericObjectsTask(bazingaTaskManager, yt, importDataRowFetchCount);
    }

    @Bean
    public CalculateAddressesDiffTask calculateAddressesDiffTask(
            Yt yt, @Qualifier("ytOperationRetryPolicy") RetryPolicy ytRetryPolicy,
            @Value("${dataapi-worker.extracted-addresses-diff.yt.delete-period-days}") int deleteDays)
    {
        CalculateAddressesDiffRunner runner = new CalculateAddressesDiffRunner(yt, ytRetryPolicy,
                YtPathsUtils.getHomeYPath("addressesDiff"), Period.days(deleteDays));

        return new CalculateAddressesDiffTask(runner);
    }
}
