package ru.yandex.chemodan.app.dataapi.worker.importer.processors;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils;
import ru.yandex.chemodan.app.dataapi.worker.importer.ImportDataChunkTask;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ImportDataYtReader;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.ypath.ImportDataYPathManager;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public abstract class BaseImportDataProcessor implements ImportDataProcessor {
    private static final Logger logger = LoggerFactory.getLogger(ImportDataYtReader.class);

    private static final String IMPORTS_YT_PATH_PREFIX = "imports";

    private final BazingaTaskManager bazingaTaskManager;

    private final ImportDataYPathManager importDataYPathManager;
    private final int importDataRowFetchCount;

    private final Yt yt;

    public BaseImportDataProcessor(BazingaTaskManager bazingaTaskManager,
            ImportDataYPathManager importDataYPathManager, int importDataRowFetchCount, Yt yt)
    {
        this.bazingaTaskManager = bazingaTaskManager;
        this.importDataYPathManager = importDataYPathManager;
        this.importDataRowFetchCount = importDataRowFetchCount;
        this.yt = yt;
    }

    @Override
    public void processImportData(String app, String dbId, Option<String> collectionId) {
        Option<YPath> pathO = getYPath(app, dbId, collectionId);
        if (!pathO.isPresent()) {
            logger.info("No data to import for today");
            return;
        }

        YPath path = pathO.get();
        logger.info("Scheduling import data from path: " + path);

        int recordCount = getRecordCount(path);

        for (int i = 0; i <= recordCount / importDataRowFetchCount; i++) {
            int lowerRowIndex = i * importDataRowFetchCount;
            int upperRowIndex = Math.min((i + 1) * importDataRowFetchCount, recordCount);

            if (lowerRowIndex < upperRowIndex) {
                ImportDataChunkTask task = ImportDataChunkTask.staticTyped(
                        lowerRowIndex, upperRowIndex, path.toString(),
                        CollectionRef.cons(new AppDatabaseRef(app, dbId), collectionId.getOrElse(getDefaultCollectionId())));
                bazingaTaskManager.schedule(task);
            }
        }

        importDataYPathManager.updateLastSuccessfulImportDateIfNeeded(app, dbId, collectionId, path.name());
    }

    protected abstract String getDefaultCollectionId();

    protected YPath getDefaultImportFolder(CollectionRef collection) {
        return YtPathsUtils.getProperYPath(IMPORTS_YT_PATH_PREFIX,
                collection.dbRef(), Option.of(collection.collectionId));
    }

    private int getRecordCount(YPath path) {
        YPath recordCountYPath = path.attribute("row_count");
        return yt.cypress().get(recordCountYPath).intValue();
    }

    private Option<YPath> getYPath(String app, String dbId, Option<String> collectionId) {
        YPath defaultImportFolder = getDefaultImportFolder(getCollection(app, dbId, collectionId));
        ListF<YPath> ypathsToImportFrom = importDataYPathManager
                .getYPathsToImportFrom(app, dbId, collectionId, defaultImportFolder);

        if (ypathsToImportFrom.size() > 1) {
            logger.info("There are more than one table in yt to import from. Using the youngest, " + ypathsToImportFrom);
        }

        return ypathsToImportFrom.lastO();
    }

    private CollectionRef getCollection(String app, String dbId, Option<String> collectionId) {
        return CollectionRef.cons(new AppDatabaseRef(app, dbId),
                collectionId.getOrElse(getDefaultCollectionId()));
    }
}
