package ru.yandex.chemodan.app.dataapi.worker.importer.readers;

import java.util.List;
import java.util.function.Consumer;

import com.fasterxml.jackson.databind.JsonNode;
import net.jodah.failsafe.RetryPolicy;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.chemodan.util.yt.JacksonTableEntryTypeWithoutUtfEncoding;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class ImportDataYtReader<R> implements ImportDataReader {
    private static final Logger logger = LoggerFactory.getLogger(ImportDataYtReader.class);

    private final Yt yt;

    private final RetryPolicy ytRetryPolicy;

    private final DataRowParser<JsonNode, R> dataRowParser;
    private final DeltaConstructor<R> deltaConstructor;

    public ImportDataYtReader(Yt yt,
            RetryPolicy ytRetryPolicy,
            DataRowParser<JsonNode, R> dataRowParser,
            DeltaConstructor<R> deltaConstructor)
    {
        this.yt = yt;
        this.ytRetryPolicy = ytRetryPolicy;
        this.dataRowParser = dataRowParser;
        this.deltaConstructor = deltaConstructor;
    }

    @Override
    public void forEachDiff(YPath path, CollectionRef collection, Function2V<DataApiUserId, Delta> callback) {
        new RetryManager()
                .withRetryPolicy(ytRetryPolicy)
                .withLogging("Getting rows from table " + path.toString())
                .runSafe(() -> yt.tables().read(
                        path,
                        new JacksonTableEntryTypeWithoutUtfEncoding(),
                        getAddressChangeConsumer(collection, callback)));
    }

    private Consumer<JsonNode> getAddressChangeConsumer(
            CollectionRef collection, Function2V<DataApiUserId, Delta> callback)
    {
        return row -> {
            try {
                R dataRow = dataRowParser.parse(row);
                Option<Tuple2<DataApiUserId, List<Delta>>> deltaForUserO = deltaConstructor.construct(collection, dataRow);
                if (deltaForUserO.isPresent()) {
                    List<Delta> deltas = deltaForUserO.get()._2;
                    deltas.forEach(d -> callback.apply(deltaForUserO.get()._1, d));
                }
            } catch (Exception e) {
                logger.error("Failed to read and properly process data row: " + row, e);
            }
        };
    }
}
