package ru.yandex.chemodan.app.dataapi.worker.importer.readers.address;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.apps.profile.FieldChangeUtils;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.Address;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 * @author vpronto
 * <p>
 * current refused:
 * - new Address -> copy current with refused_
 * - Address exists -> copy current with refused_; delete refused_ insert common
 * <p>
 * current deny_suggest:
 * - new Address -> copy current with refused_; set deny_suggest to new one
 * - Address exists -> copy current with refused_; delete refused_;  set deny_suggest to new one
 */
public class AddressChangeConstructor {

    private final static Logger logger = LoggerFactory.getLogger(AddressChangeConstructor.class);

    public static final ListF<String> FIELDS_TO_COMPARE = Cf.list(Address.ADDRESS_LINE, Address.ADDRESS_LINE_SHORT,
            Address.LATITUDE, Address.LONGITUDE);

    public static ListF<RecordChange> constructChange(CollectionRef collection,
                                                      MapF<String, Address> addressMap, AddressChange data, long rev)
    {

        Optional<Address> existsSameRefused = findSameRefused(addressMap, data.address);
        Option<Address> updatedAddress = addressMap.getO(data.address.recordId());
        logger.debug("About to process: {}, existsSameRefused {},  address to update: {}",
                data, existsSameRefused, updatedAddress);
        if (!updatedAddress.isPresent()) {
            return processNewAddress(existsSameRefused, collection, data);
        } else {
            return processExistsAddress(updatedAddress, existsSameRefused, data, rev, collection);
        }
    }

    private static ListF<RecordChange> processExistsAddress(Option<Address> updatedAddress, Optional<Address> refused,
                                                            AddressChange data, long rev, CollectionRef collection)
    {
        ListF<RecordChange> changes = new ArrayListF<>();
        Address newAddress = data.address;
        Address currentAddress = updatedAddress.get();
        Set<AddressTag> tags = findTags(currentAddress.getTags());
        switch (data.addressChangeType) {
            case REMOVE:
                if (tags.contains(AddressTag.DENY_SUGGEST)) {
                    logger.debug("Skip remove for deny");
                    return changes;
                } else if (tags.contains(AddressTag.REFUSED)) {
                    logger.debug("Move current to refused: {}", currentAddress);
                    changes.add(copyAddressWithRefused(currentAddress, collection, rev));
                }
                changes.add(RecordChange.delete(collection, newAddress.recordId()));
                return changes;
            case ADD:
            case CHANGE:
                if (!tags.isEmpty()) {
                    if (tags.contains(AddressTag.DENY_SUGGEST)) {
                        logger.debug("processing  DENY_SUGGEST");
                        newAddress = data.address.withTags(Cf.list(AddressTag.DENY_SUGGEST.value()));
                    } else if (tags.contains(AddressTag.REFUSED)) {
                        logger.debug("processing  REFUSED");
                        newAddress = data.address.withTags(Cf.list(AddressTag.REFUSED.value()));
                    }
                    changes.addAll(processRefused(currentAddress, newAddress, refused, collection, rev));
                }
                Option<RecordChange> recordChanges = processChange(currentAddress, newAddress, collection);
                recordChanges.ifPresent(recordChange -> changes.add(recordChange));
                return changes;
        }
        return changes;
    }


    private static ListF<RecordChange> processNewAddress(Optional<Address> refused, CollectionRef collection,
                                                         AddressChange data)
    {
        ListF<RecordChange> changes = new ArrayListF<>();

        switch (data.addressChangeType) {
            case ADD:
            case CHANGE:
                if (refused.isPresent()) {
                    logger.debug("Moving refused to main: {}", refused.get());
                    RecordChange delete = RecordChange.delete(collection, refused.get().recordId());
                    changes.add(delete);
                }
                changes.add(data.address.toFieldSet(collection).toInsertChange());
                break;
            case REMOVE:
                break;
        }
        return changes;
    }

    private static Option<RecordChange> processChange(Address currentAddress, Address newAddress,
                                                      CollectionRef collection)
    {
        updateDateTimeFields(currentAddress, newAddress);
        ListF<FieldChange> fieldChanges = filterChanges(
                FieldChangeUtils.createFieldChanges(currentAddress, newAddress));
        if (!fieldChanges.isEmpty()) {
            return Option.of(RecordChange.update(collection, newAddress.recordId(), fieldChanges));
        } else {
            return Option.empty();
        }
    }

    private static ListF<RecordChange> processRefused(Address currentAddress, Address newAddress,
                                                      Optional<Address> refused, CollectionRef collection,
                                                      long rev)
    {
        ListF<RecordChange> changes = new ArrayListF<>();
        if (!checkSame(currentAddress, newAddress)) {
            logger.debug("copy currentAddress: {}", currentAddress);
            RecordChange recordChanges = copyAddressWithRefused(currentAddress, collection, rev);
            changes.add(recordChanges);
        }
        refused.ifPresent(address1 -> {
            if (!refused.get().recordId().equals(currentAddress.recordId())) {
                logger.debug("new Address exists in refused list and not main -> delete it: {}", address1);
                RecordChange delete = RecordChange.delete(collection, address1.recordId());
                changes.add(delete);
            }
        });
        return changes;
    }

    private static boolean checkSame(Address currentAddress, Address address) {
        return address.getAddressLine().equals(currentAddress.getAddressLine()) &&
                address.getAddressLineShort().equals(currentAddress.getAddressLineShort());
    }

    private static RecordChange copyAddressWithRefused(Address currentAddress, CollectionRef collection, long rev) {
        Address copyAddress = currentAddress.withId(currentAddress.recordId() +
                        "_" + AddressTag.REFUSED.value() + "_" + rev);
        if (!currentAddress.getTags().containsF().apply(AddressTag.REFUSED.value())) {
            copyAddress = copyAddress.withTags(currentAddress.getTags().plus(AddressTag.REFUSED.value()));
        }
        return copyAddress.toFieldSet(collection).toInsertChange();
    }

    private static Optional<Address> findSameRefused(MapF<String, Address> addressMap, Address currentAddress) {
        return addressMap.values().stream()
                .filter((Function1B<Address>) address -> address.getTags().containsF()
                        .apply(AddressTag.REFUSED.value()))
                .filter(a -> checkSame(a, currentAddress))
                .findFirst();
    }

    private static Set<AddressTag> findTags(ListF<String> tags) {
        return tags.stream()
                .map(AddressTag.R::fromValueO)
                .filter(Option::isPresent)
                .map(Option::get)
                .collect(Collectors.toSet());
    }


    private static void updateDateTimeFields(Address currentAddress, Address newAddress) {
        newAddress.setCreated(currentAddress.getCreated());
        newAddress.setLastUsed(currentAddress.getLastUsed());
    }

    private static ListF<FieldChange> filterChanges(ListF<FieldChange> fieldChanges) {
        ListF<FieldChange> result = fieldChanges.filter(fieldChange -> FIELDS_TO_COMPARE.containsTs(fieldChange.key));
        return result.isEmpty() ? result : fieldChanges;
    }
}
