package ru.yandex.chemodan.app.dataapi.worker.importer.readers.address;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.Address;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.test.Assert;

/**
 * @author metal
 */
public class AddressChangeConstructorTest {
    private static final String ADDRESS_ID = "home";

    private static final CollectionRef COLLECTION =
            CollectionRef.cons(new AppDatabaseRef("testapp", "testdb"), "testcol");

    private Address address;

    @Before
    public void init() {
        double lat = 12.0;
        double lon = 24.0;
        Option<String> addressLine = Option.of("addr");
        Option<String> addressLineShort = Option.of("addr_short");
        Instant time = Instant.now();

        address = createAddress(lat, lon, addressLine, addressLineShort, time);
    }

    @Test
    public void constructEmptyChange() {
        ListF<RecordChange> changes = AddressChangeConstructor.constructChange(COLLECTION, Cf.map(),
                createAddressChange(address, AddressChangeType.REMOVE), 1);
        Assert.assertTrue(changes.isEmpty());

        changes = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(copyAddress(address), AddressChangeType.ADD), 1);
        Assert.assertTrue(changes.isEmpty());

        changes = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(copyAddress(address), AddressChangeType.CHANGE), 1);
        Assert.assertTrue(changes.isEmpty());
    }

    @Test
    public void constructChangeWithEmptyAddressInDatabase() {
        ListF<RecordChange> change = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(), createAddressChange(address, AddressChangeType.ADD), 1);
        change.forEach(c -> checkChangeForContainingFullAddress(c));

        change = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(), createAddressChange(address, AddressChangeType.CHANGE), 1);
        change.forEach(c -> checkChangeForContainingFullAddress(c));
    }

    @Test
    public void constructChangeWithAddressInDatabase() {
        Address copiedAddress = copyAddress(address);
        copiedAddress.setAddressLineShort(Option.of("new address short"));

        ListF<RecordChange> changes = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(copiedAddress, AddressChangeType.ADD), 1);
        Assert.assertFalse(changes.isEmpty());
        changes.forEach(d -> checkChangeForSpecificAddressUpdate(d));

        changes = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(copiedAddress, AddressChangeType.CHANGE), 1);
        Assert.assertFalse(changes.isEmpty());
        changes.forEach(d -> checkChangeForSpecificAddressUpdate(d));
    }

    @Test
    public void constructChangeThatRemovesAddress() {
        ListF<RecordChange> change = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(address, AddressChangeType.REMOVE), 1);
        change.forEach(c -> {
            Assert.equals(RecordChangeType.DELETE, c.type);
            Assert.equals(ADDRESS_ID, c.recordId);
            Assert.assertTrue(c.getFields().isEmpty());
        });
    }

    @Test
    public void constructRefusedChangeThatChangeAddress() {
        Address address = this.address.withTags(Cf.list(AddressTag.REFUSED.value()));

        Address copiedAddress = copyAddress(address);
        copiedAddress.setAddressLineShort(Option.of("new address short"));

        ListF<RecordChange> change = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(copiedAddress, AddressChangeType.CHANGE), 1);
        RecordChange insertChange = change.get(0);
        Assert.equals(RecordChangeType.INSERT, insertChange.type);
        Assert.assertContains(insertChange.recordId, "refused_");

        RecordChange updateChange = change.get(1);
        checkChangeForSpecificAddressUpdate(updateChange);
    }

    @Test
    public void constructRefusedChangeThatRemove() {
        Address address = this.address.withTags(Cf.list(AddressTag.REFUSED.value()));

        ListF<RecordChange> change = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(address.recordId(), address), createAddressChange(address, AddressChangeType.REMOVE), 1);
        RecordChange insertChange = change.get(0);
        Assert.equals(RecordChangeType.INSERT, insertChange.type);
        Assert.assertContains(insertChange.recordId, "refused_");

        RecordChange removeChange = change.get(1);
        Assert.equals(RecordChangeType.DELETE, removeChange.type);
        Assert.equals(ADDRESS_ID, removeChange.recordId);
        Assert.assertTrue(removeChange.getFields().isEmpty());
    }

    @Test
    public void constructRefusedChangeAlreadyExists() {

        Address deny = copyAddress(address).withTags(Cf.list(AddressTag.DENY_SUGGEST.value()));

        Address alreadyRefused = deny.withId("already_refused").withTags(Cf.list(AddressTag.REFUSED.value()));
        alreadyRefused.setAddressLineShort(Option.of("new address short"));
        Address newAddress = copyAddress(deny).withId(address.recordId());
        newAddress.setAddressLineShort(Option.of("new address short"));

        ListF<RecordChange> change = AddressChangeConstructor.constructChange(COLLECTION,
                Cf.map(deny.recordId(), deny, alreadyRefused.recordId(), alreadyRefused),
                createAddressChange(newAddress, AddressChangeType.CHANGE), 1);
        Assert.equals(3, change.size());
    }

    private void checkChangeForSpecificAddressUpdate(RecordChange change) {
        Assert.equals(RecordChangeType.UPDATE, change.type);
        Assert.equals(ADDRESS_ID, change.recordId);
        Assert.assertEquals(2, change.getFields().size());
        checkChangeForContainingSpecificFields(change, Cf.list(Address.ADDRESS_LINE_SHORT, Address.MODIFIED));
    }

    private void checkChangeForContainingFullAddress(RecordChange change) {
        Assert.equals(RecordChangeType.INSERT, change.type);
        Assert.equals(ADDRESS_ID, change.recordId);
        Assert.assertEquals(address.toDataMap().size(), change.getFields().size());
        checkChangeForContainingSpecificFields(change, AddressChangeConstructor.FIELDS_TO_COMPARE);
    }

    private void checkChangeForContainingSpecificFields(RecordChange change, ListF<String> fieldNames) {
        fieldNames.forEach(fieldName -> checkChangeForContainingSpecificField(change, fieldName));
    }

    private void checkChangeForContainingSpecificField(RecordChange change, String fieldName) {
        Assert.assertFalse(change.getFields()
                .filter(localFieldId -> localFieldId.fieldId.equals(fieldName))
                .isEmpty());
    }

    private AddressChange createAddressChange(Address address, AddressChangeType type) {
        return new AddressChange(DataApiUserId.parse("123"), address, type);
    }

    private Address createAddress(double lat, double lon,
            Option<String> addressLine, Option<String> addressLineShort, Instant time)
    {
        return new Address(Option.of(ADDRESS_ID), "", new Coordinates(lat, lon), addressLine, addressLineShort,
                Option.of(time), Option.of(time), Option.of(time), Cf.list(), Cf.list(), Option.empty(), Option.empty());
    }

    private Address copyAddress(Address address) {
        Address copiedAddress = Address.fromData(address.getAddressId().get(), address.toDataMap());
        Instant time = Instant.now().plus(1000);
        copiedAddress.setCreated(Option.of(time));
        copiedAddress.setModified(Option.of(time));
        copiedAddress.setLastUsed(Option.of(time));
        return copiedAddress;
    }
}
