package ru.yandex.chemodan.app.dataapi.worker.importer.readers.address;

import java.util.List;

import net.jodah.failsafe.RetryPolicy;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.address.Address;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.worker.importer.readers.DeltaConstructor;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class AddressDeltaConstructor implements DeltaConstructor<AddressChange> {
    private static final Logger logger = LoggerFactory.getLogger(AddressDeltaConstructor.class);

    private final DataApiManager dataApiManager;

    private final RetryPolicy dbRetryPolicy;

    public AddressDeltaConstructor(DataApiManager dataApiManager, RetryPolicy dbRetryPolicy) {
        this.dataApiManager = dataApiManager;
        this.dbRetryPolicy = dbRetryPolicy;
    }

    @Override
    public Option<Tuple2<DataApiUserId, List<Delta>>> construct(CollectionRef collection, AddressChange data) {
        Option<Database> databaseO = getOrCreateDatabase(data.uid, collection);
        if (!databaseO.isPresent()) {
            logger.error("User {} doesn't have database {}, skipping.", data.uid, getCollectionFullName(collection));
            return Option.empty();
        }

        UserDatabaseSpec dbSpec = databaseO.get().spec();
        MapF<String, Address> addressMapF = dataApiManager
                .getRecords(dbSpec).map(Address::fromDataRecord)
                .toMap(m -> new Tuple2<>(m.getAddressId().get(), m));

        ListF<RecordChange> recordChanges = AddressChangeConstructor
                .constructChange(collection, addressMapF, data, databaseO.get().rev);
        return Option.of(Tuple2.tuple(data.uid, recordChanges.map(recordChange -> recordChange.toDelta())));
    }

    private Option<Database> getOrCreateDatabase(DataApiUserId uid, CollectionRef collection) {
        return new RetryManager<Database>()
                .withRetryPolicy(dbRetryPolicy)
                .withLogging("Getting or creating database " + getCollectionFullName(collection))
                .getSafe(() -> doGetOrCreateDatabase(uid, collection));
    }

    private Database doGetOrCreateDatabase(DataApiUserId uid, CollectionRef collection) {
        return dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(uid, collection.dbRef()));
    }

    private String getCollectionFullName(CollectionRef collection) {
        return collection.dbRef().dbAppId() + "." + collection.dbRef().databaseId() + "." + collection.collectionId;
    }
}
