package ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic;

import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.Listeners;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.SyncFailsafe;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class GenericObjectChangeApplier {
    private static final Logger logger = LoggerFactory.getLogger(GenericObjectChangeApplier.class);

    private final DataApiManager dataApiManager;
    private final SyncFailsafe failsafe;

    public GenericObjectChangeApplier(DataApiManager dataApiManager, RetryPolicy retryPolicy) {
        this.dataApiManager = dataApiManager;

        Listeners<?> listeners = new Listeners<>().onFailedAttempt((r, e, ctx) ->
                logger.warn("Failed to perform action#{}: {}", ctx.getExecutions(), ExceptionUtils.getAllMessages(e)));

        this.failsafe = Failsafe.with(retryPolicy).with(listeners);
    }

    public Option<ApplyResult> applySafeWithRetries(GenericObjectChange change, TypeLocation location) {
        try {
            return Option.of(applyWithRetries(change, location));

        } catch (RuntimeException e) {
            logger.warn("Failed to apply changes for {}: {}", change.uid, ExceptionUtils.getAllMessages(e));
            return Option.empty();
        }
    }

    public ApplyResult applyWithRetries(GenericObjectChange change, TypeLocation location) {
        return failsafe.get(() -> apply(change, location));
    }

    public ApplyResult apply(GenericObjectChange change, TypeLocation location) {
        RecordId recId = location.toColRef().consRecordRef(change.recordId).id();

        if (change.operation == ChangeOperation.INSERT || change.operation == ChangeOperation.REPLACE) {
            Database db = dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(change.uid, location.dbRef()));
            Option<DataRecord> rec = dataApiManager.getRecord(db.spec(), recId);

            if (!rec.isPresent()) {
                dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                        new Delta(RecordChange.insert(recId, change.recordData)));

                return ApplyResult.CREATED;

            } else if (change.operation == ChangeOperation.REPLACE && !rec.get().getData().equals(change.recordData)) {
                dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                        new Delta(RecordChange.set(recId, change.recordData)));

                return ApplyResult.UPDATED;
            }

        } else if (change.operation == ChangeOperation.DELETE) {
            Option<Database> dbO = dataApiManager.getDatabaseO(new UserDatabaseSpec(change.uid, location.dbRef()));
            Option<DataRecord> recO = dbO.flatMapO(db -> dataApiManager.getRecord(db.spec(), recId));

            if (recO.isPresent()) {
                dataApiManager.applyDelta(dbO.get(), RevisionCheckMode.PER_RECORD, new Delta(RecordChange.delete(recId)));

                return ApplyResult.DELETED;
            }
        }
        return ApplyResult.SKIPPED;
    }
}
