package ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic;

import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class GenericObjectChangeParser {
    private final Logger logger = LoggerFactory.getLogger(GenericObjectChangeParser.class);

    private final TypeSettings type;
    private final FormatConverter converter;

    public GenericObjectChangeParser(TypeSettings type, FormatConverter converter) {
        this.type = type;
        this.converter = converter;
    }

    public Option<GenericObjectChange> parseSafe(JsonNode row) {
        try {
            return Option.of(parse(row));

        } catch (RuntimeException e) {
            logger.warn("Failed to parse object change: {}", ExceptionUtils.getAllMessages(e));
            return Option.empty();
        }
    }

    public GenericObjectChange parse(JsonNode row) {
        DataApiUserId uid = DataApiUserId.parse(row.get("uid").asText("Missing uid"));

        ChangeOperation operation = ChangeOperation.R.valueOf(
                StringUtils.notEmptyO(row.path("operation").asText()).getOrElse(ChangeOperation.REPLACE.name()));

        MapF<String, DataField> data = Cf.map();

        if (operation != ChangeOperation.DELETE) {
            JsonNode node = row.path("record_data");

            data = node.isObject() ? converter.toDataFields(node) : converter.toDataFields(node.asText("Missing data"));
        }

        Option<String> idO = data.getO(type.idPropertyName).map(DataField::stringValue);

        String recordId = StringUtils.notEmptyO(row.path("record_id").asText(null)).orElse(idO).getOrThrow("Missing id");

        data = data.filterKeys(k -> !type.idPropertyName.equals(k));

        return new GenericObjectChange(uid, recordId, operation, data);
    }
}
