package ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic;

import java.util.NoSuchElementException;

import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
public class GenericObjectChangeParserTest {

    private static final FormatConverter converter = new FormatConverter(
            "{"
            + "  \"type\":\"object\","
            + "  \"properties\": {"
            + "    \"string\": {\"type\": \"string\"},"
            + "    \"number\": {\"type\": \"integer\"}"
            + "  }"
            + "}");

    private static final TypeSettings type = new TypeSettings(
            "", "", "string", false, false, Cf.list(), Mockito.mock(TypeLocation.class), Option.empty(), false);

    private static final GenericObjectChangeParser parser = new GenericObjectChangeParser(type, converter);

    @Test
    public void textualData() {
        GenericObjectChange change = parser.parse(JsonNodeUtils.getNode(
                "{\"uid\": 1, \"record_id\": \"id\", \"record_data\": \"{\\\"number\\\": 29}\"}"));

        Assert.equals(DataField.integer(29), change.recordData.getOrThrow("number"));
    }

    @Test
    public void dataProvidedId() {
        GenericObjectChange change = parser.parse(JsonNodeUtils.getNode(
                "{\"uid\": 1, \"record_data\": {\"string\": \"id\", \"number\": 7}}"));

        Assert.equals("id", change.recordId);
        Assert.hasSize(1, change.recordData);

        Assert.equals(ChangeOperation.REPLACE, change.operation);
        Assert.equals(DataApiUserId.parse("1"), change.uid);
    }

    @Test
    public void dataMissingDeletion() {
        GenericObjectChange change = parser.parse(JsonNodeUtils.getNode(
                "{\"uid\": 1, \"record_id\": \"id\", \"operation\": \"delete\"}"));

        Assert.equals("id", change.recordId);
        Assert.isEmpty(change.recordData);

        Assert.equals(ChangeOperation.DELETE, change.operation);
    }

    @Test
    public void missingId() {
        Function<String, GenericObjectChange> parse = json -> parser.parse(JsonNodeUtils.getNode(json));

        Assert.assertThrows(() -> parse.apply("{\"uid\": 3, \"operation\": \"delete\"}"),
                NoSuchElementException.class, e -> e.getMessage().contains("Missing id"));

        Assert.assertThrows(() -> parse.apply("{\"uid\": 2, \"record_id\": \"\", \"operation\": \"delete\"}"),
                NoSuchElementException.class, e -> e.getMessage().contains("Missing id"));

        Assert.assertThrows(() -> parse.apply("{\"uid\": 1, \"record_id\": null, \"operation\": \"delete\"}"),
                NoSuchElementException.class, e -> e.getMessage().contains("Missing id"));
    }
}
