package ru.yandex.chemodan.app.dataapi.worker.importer.readers.generic;

import net.jodah.failsafe.RetryPolicy;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

/**
 * @author dbrylev
 */
public class GenericObjectsProcessor {
    private final DataApiManager dataApiManager;
    private final RetryPolicy dbRetryPolicy;

    private final Yt yt;
    private final RetryPolicy ytRetryPolicy;

    public GenericObjectsProcessor(
            DataApiManager dataApiManager, RetryPolicy dbRetryPolicy, Yt yt, RetryPolicy ytRetryPolicy)
    {
        this.dataApiManager = dataApiManager;
        this.dbRetryPolicy = dbRetryPolicy;
        this.yt = yt;
        this.ytRetryPolicy = ytRetryPolicy;
    }

    public void processAndWriteResult(TypeSettings type, YPath sourcePath, ProcessingContext context, YPath resultPath) {
        ProcessingStats stats = new ProcessingStats(context);

        process(type, sourcePath, stats);

        new RetryManager()
                .withRetryPolicy(ytRetryPolicy)
                .withLogging("Writing result to table " + resultPath.toString())
                .runSafe(() -> yt.tables().write(
                        resultPath, YTableEntryTypes.bender(ProcessingStats.class), Cf.list(stats)));
    }

    public void process(TypeSettings type, YPath path, ProcessingStats stats) {
        GenericObjectChangeParser parser = new GenericObjectChangeParser(type, new FormatConverter(type.jsonSchema));
        GenericObjectChangeApplier applier = new GenericObjectChangeApplier(dataApiManager, dbRetryPolicy);

        new RetryManager()
                .withRetryPolicy(ytRetryPolicy)
                .withLogging("Getting rows from table " + path.toString())
                .runSafe(() -> yt.tables().read(path, YTableEntryTypes.JACKSON_UTF8, node -> {
                    Option<GenericObjectChange> change = parser.parseSafe(node);

                    if (change.isPresent()) {
                        Option<ApplyResult> result = applier.applySafeWithRetries(change.get(), type.typeLocation);

                        if (result.isPresent()) {
                            stats.incApplied(result.get());
                        } else {
                            stats.incUnapplied();
                        }
                    } else {
                        stats.incUnparsed();
                    }
                }));
    }
}
