package ru.yandex.chemodan.app.dataapi.worker.importer.readers.ypath;

import java.util.List;
import java.util.stream.Collectors;

import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils.YT_NODE_NAME_FORMATTER;

/**
 * @author metal
 */
public class ImportDataYPathManager {
    private static final Logger logger = LoggerFactory.getLogger(ImportDataYPathManager.class);

    private final LastSuccessfulImportDateRegistry lastSuccessfulImportDateRegistry;
    private final ImportYPathOverridesRegistry importYPathOverridesRegistry;
    private final Yt yt;

    public ImportDataYPathManager(LastSuccessfulImportDateRegistry lastSuccessfulImportDateRegistry,
            ImportYPathOverridesRegistry importYPathOverridesRegistry, Yt yt)
    {
        this.lastSuccessfulImportDateRegistry = lastSuccessfulImportDateRegistry;
        this.importYPathOverridesRegistry = importYPathOverridesRegistry;
        this.yt = yt;
    }

    public ListF<YPath> getYPathsToImportFrom(CollectionRef collection, YPath defaultFolderYPath) {
        return getYPathsToImportFrom(collection.dbAppId(), collection.databaseId(), Option.of(collection.collectionId),
                defaultFolderYPath);
    }

    public ListF<YPath> getYPathsToImportFrom(String appId, String databaseId, Option<String> collectionId,
            YPath defaultFolderYPath)
    {
        Option<String> forcedYPath = importYPathOverridesRegistry
                .getForcedImportDataFullPath(appId, databaseId, collectionId);
        if (forcedYPath.isPresent()) {
            logger.info("Import path is forced to: " + forcedYPath.get());
            return Cf.list(YPath.simple(forcedYPath.get()));
        }

        Option<String> importDataFolderPathO = importYPathOverridesRegistry
                .getImportDataFolderPath(appId, databaseId, collectionId);
        YPath importDataFolderPath = importDataFolderPathO.map(YPath::simple).getOrElse(defaultFolderYPath);

        DateTime lastSuccessfulDate = YT_NODE_NAME_FORMATTER.parseDateTime(lastSuccessfulImportDateRegistry
                .getLastSuccessfulImportDate(appId, databaseId, collectionId));

        logger.info("Trying to get yt path of table to import from: " +
                "[folder=" + importDataFolderPath +
                ", last_successful_import=" + YT_NODE_NAME_FORMATTER.print(lastSuccessfulDate) + "]");

        List<String> datesToImport = yt.cypress()
                .list(importDataFolderPath)
                .stream()
                .map(YTreeStringNode::getValue)
                .filter(date -> {
                    try {
                        DateTime dateTime = YT_NODE_NAME_FORMATTER.parseDateTime(date);
                        return dateTime.compareTo(lastSuccessfulDate) > 0;
                    } catch (IllegalArgumentException e) {
                        return false;
                    }
                })
                .sorted()
                .collect(Collectors.toList());

        return Cf.wrap(datesToImport).map(importDataFolderPath::child);
    }

    public void updateLastSuccessfulImportDateIfNeeded(String app, String dbId, Option<String> collectionId,
            String date)
    {
        if (!importYPathOverridesRegistry.getForcedImportDataFullPath(app, dbId, collectionId).isPresent()) {
            lastSuccessfulImportDateRegistry.updateLastSuccessfulImportDate(app, dbId, collectionId, date);
        }
    }
}
