package ru.yandex.chemodan.notifier;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserType;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
@AllArgsConstructor
public class NotifierUnreadCountProvider {

    private static final Logger logger = LoggerFactory.getLogger(NotifierUnreadCountProvider.class);

    private final DataApiManager dataApiManager;
    private final NotifierEnabledServicesProvider enabledServicesProvider;


    public int getForEnabledServices(DataApiUserId uid) {
        ListF<String> enabledServices = uid.type == DataApiUserType.YT
                ? enabledServicesProvider.getEnabledYaTeamServiceNames()
                : enabledServicesProvider.getEnabledServiceNames();

        ListF<String> enabledRecordIds = enabledServices.map(s -> NotifierDatabasing.metaRecordIdFor(s).recordId());

        ListF<DataRecord> records = dataApiManager.getRecords(
                new UserDatabaseSpec(uid, NotifierDatabasing.NOTIFICATIONS_DB),
                RecordsFilter.DEFAULT.withCollectionId(NotifierDatabasing.META_COLLECTION));

        records = records.filter(enabledRecordIds.containsF().compose(DataRecord::getRecordId));

        return records.map(this::extractCount).sum(Cf.Integer);
    }

    public int getForService(DataApiUserId uid, String service) {
        return getForService(new UserDatabaseSpec(uid, NotifierDatabasing.NOTIFICATIONS_DB), service);
    }

    public int getForService(Database notificationsDb, String service) {
        return getForService(UserDatabaseSpec.fromDatabase(notificationsDb), service);
    }

    private int getForService(UserDatabaseSpec database, String service) {
        Option<DataRecord> record = dataApiManager.getRecord(
                database, NotifierDatabasing.metaRecordIdFor(service));

        return record.map(this::extractCount).getOrElse(0);
    }

    private int extractCount(DataRecord record) {
        DataField df = record.getData().getTs(NotifierDatabasing.UNREAD_COUNT_FIELD);

        switch (df.fieldType) {
            case INTEGER:
                return df.integerValue().intValue();

            case DECIMAL:
                // hack for failsafe retrieval in case of database tampering
                logger.warn("Field " + NotifierDatabasing.UNREAD_COUNT_FIELD + " should be long: {}", df);
                return (int) Math.round(df.decimalValue());

            default:
                logger.error("Can't cast " + NotifierDatabasing.UNREAD_COUNT_FIELD + " field to number: {}", df);
                return 0;
        }
    }
}
