package ru.yandex.chemodan.app.dataapi.worker;

import java.util.Collections;

import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;
import org.junit.Assert;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.web.direct.unmarshallers.DataFieldJsonParsers;
import ru.yandex.chemodan.app.dataapi.web.direct.unmarshallers.DirectPojos;
import ru.yandex.chemodan.util.bender.ISOInstantUnmarshaller;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
@Setter
@ToString
public class RestoreManager {

    private static final Logger logger = LoggerFactory.getLogger(RestoreManager.class);
    private static final BenderParser<RestoreManager.Data> parserYtData = Bender.parser(RestoreManager.Data.class);

    public final DataApiManager dataApiManager;
    public final DatabasesJdbcDao databasesJdbcDao;

    public static String DS_APP = "maps_common";
    public static String DB_ID = "ymapsbookmarks1";
    public static String COLLECTION = "bookmarks";

    public static final DatabaseRef DATABASE_REF = DatabaseRef.cons(Option.of(DS_APP), DB_ID);

    public Database findOrCreateDb(DataApiUserId uid) {
        return findDb(uid).getOrElse(() -> initInternal(uid));
    }

    public Option<Database> findDb(DataApiUserId uid) {
        return dataApiManager.getDatabaseO(dbSpec(uid));
    }

    public UserDatabaseSpec dbSpec(DataApiUserId uid) {
        return new UserDatabaseSpec(uid, DATABASE_REF);
    }

    public Database initInternal(DataApiUserId uid) {
        Database db = dataApiManager.getOrCreateDatabase(dbSpec(uid));
        return db;
    }

    public MapF<DataApiUserId, Snapshot> restoreListOfUsers(ListF<String> users) {
        MapF<DataApiUserId, Snapshot> result = Cf.linkedHashMap();
        users.forEach(s -> {
            DataCompacted compacted = new DataCompacted();
            compacted.setUserId(DataApiUserId.parse(s));
            Option<Snapshot> snapshots = fixUser(compacted);
            snapshots.ifPresent(snapshot -> result.put(compacted.getUserId(), snapshot));
        });
        return result;
    }

    public MapF<DataApiUserId, Snapshot> restoreUsers(String path) {
        MapF<DataApiUserId, Snapshot> result = Cf.linkedHashMap();
        logger.info("About to restore users using file: {}", path);
        MapF<DataApiUserId, MapF<String, ListF<RestoreManager.Data>>> rawData = RestoreManager.loadData(path);
        logger.info("Loaded {} users", rawData.size());
        MapF<DataApiUserId, RestoreManager.DataCompacted> dataApiUserIdDataCompactedMapF = genDeltas(rawData);
        for (RestoreManager.DataCompacted compacted : dataApiUserIdDataCompactedMapF.values()) {

            Option<Snapshot> snapshots = fixUser(compacted);
            snapshots.ifPresent(s -> result.put(compacted.getUserId(), s));
        }
        return result;
    }

    public Option<Snapshot> fixUser(DataCompacted compacted) {
        Database db = findOrCreateDb(compacted.getUserId());
        Snapshot snapshotBefore = dataApiManager.getSnapshot(db.spec());
        if (snapshotBefore.records.ids().size() > 0) {
            logger.info("User has data skipping uid={}, {}", compacted.getUserId(), snapshotBefore);
            return Option.empty();
        }

        logger.info("Restoring uid={}, {}", compacted.getUserId(), compacted);
        logger.info("Creating root folder uid={}, {}", compacted.getUserId(), compacted.getRootFolder());
        applyDeltas(compacted.getUserId(), compacted.getRootFolder());
        logger.info("Creating favoriteFolder folder uid={}, {}", compacted.getUserId(), compacted.getFavoriteFolder());
        applyDeltas(compacted.getUserId(), compacted.getFavoriteFolder());
        logger.info("Creating rest folders uid={}, {}", compacted.getUserId(), compacted.getDeltas());
        MapF<String, ListF<Delta>> deltas = compacted.getDeltas();
        for (ListF<Delta> deltaListF : deltas.values()) {
            applyDeltas(compacted.getUserId(), deltaListF);
        }

        logger.info("Db {} for uid={}", db, compacted.getUserId());
        Snapshot snapshot = dataApiManager.getSnapshot(db.spec());
        logger.info("Snapshot {} for uid={}", snapshot, compacted.getUserId());
        Assert.assertNotNull(snapshot);
        logger.info("uid={} restored");
        db = findOrCreateDb(compacted.getUserId());
        long rev = db.rev;
        Long newRev = rev + 3000;
        db = db.withRev(newRev);
        databasesJdbcDao.save(db, rev);
        logger.info("Db {} for uid={}", db, compacted.getUserId());
        return Option.of(snapshot);
    }

    //
    public static MapF<DataApiUserId, MapF<String, ListF<RestoreManager.Data>>> loadData(String path) {

        ListF<RestoreManager.Data> dataList = parserYtData.parseListJson(new File2(path));
        logger.info("Loaded data from path: {}, data: {}", path, dataList);
        MapF<DataApiUserId, MapF<String, ListF<RestoreManager.Data>>> map = Cf.linkedHashMap();
        for (RestoreManager.Data data : dataList) {

            map.compute(DataApiUserId.parse(data.getUid()),
                    (Function2<DataApiUserId, MapF<String, ListF<Data>>, MapF<String, ListF<Data>>>) (userId, m) -> {
                        if (m == null) {
                            m = Cf.linkedHashMap();
                        }
                        String record_id = data.getRecord_id();
                        ListF<Data> list = m.getO(record_id).getOrElse(Cf.arrayList());
                        list.add(data);
                        m.put(record_id, list);
                        return m;
                    });
        }
        for (Tuple2<DataApiUserId, MapF<String, ListF<Data>>> e : map.entries()) {
            for (ListF<Data> d : e.get2().values()) {
                Collections.sort(d);
                if (d.get(0).getMethod().equals("update")) {
                    d.get(0).setMethod("insert");
                }
            }
        }
        return map;
    }

    public static MapF<DataApiUserId, DataCompacted> genDeltas(MapF<DataApiUserId, MapF<String, ListF<RestoreManager.Data>>> map) {
        MapF<DataApiUserId, DataCompacted> deltas = Cf.linkedHashMap();
        for (Tuple2<DataApiUserId, MapF<String, ListF<Data>>> e : map.entries()) {
            Tuple2List<String, ListF<Data>> entries = e.get2().entries();
            DataCompacted compacted = new DataCompacted();
            compacted.setUserId(e.get1());
            for (Tuple2<String, ListF<Data>> entry : entries) {
                ListF<Data> dataList = entry.get2();
                if (dataList.get(dataList.size() - 1).getMethod().equals("delete")) {
                    Validate.isFalse(e.get1().equals("the_favorites_folder"));
                    Validate.isFalse(e.get1().equals("the_root_folder"));
                    continue;
                }
                ListF<Delta> deltasList = dataList.map(i -> RestoreManager.genDelta(i));
                if (entry.get1().equals("the_favorites_folder")) {
                    compacted.setFavoriteFolder(deltasList);
                } else if (entry.get1().equals("the_root_folder")) {
                    compacted.setRootFolder(deltasList);
                } else {
                    compacted.getDeltas().put(entry.get1(), deltasList);
                }
            }
            deltas.put(compacted.getUserId(), compacted);
        }

        return deltas;
    }

    public void applyDeltas(DataApiUserId userId, ListF<Delta> deltas) {
        for (Delta delta : deltas) {
            applyDelta(userId, delta);
        }
    }

    public void applyDelta(DataApiUserId userId, Delta delta) {
        try {
            logger.warn("About to apply changes: {} uid={}", delta, userId);
            Database orCreateDb = findOrCreateDb(userId);
            logger.warn("Database: {}, uid={}", orCreateDb, userId);
            Database database = dataApiManager.applyDelta(orCreateDb, RevisionCheckMode.PER_RECORD, delta);
            logger.warn("Result: {}, uid={}", database, userId);
        } catch (Exception e) {
            logger.error("Can't apply delta uid={}, {}", userId, delta, e);
        }
    }

    public static final BenderMapper innerMapper = new BenderMapper(
            new BenderConfiguration(
                    new BenderSettings(),
                    CustomMarshallerUnmarshallerFactoryBuilder.cons()
                            .add(Instant.class, new ISOInstantUnmarshaller())
                            .add(DataField.class, DataFieldJsonParsers.consUnmarshaller())
                            .build()
            )
    );

    public static Delta genDelta(RestoreManager.Data data) {
        DirectPojos.DirectSnapshotItem snapshotPojoRow = innerMapper.parseJson(DirectPojos.DirectSnapshotItem.class,
                "{\"fields\":" + data.getRecord_data().get().replaceAll("\\\\\"", "\"")
                        + ", \"revision\":" + data.record_rev
                        + ", \"collection_id\":\"" + COLLECTION
                        + "\", \"record_id\":\"" + data.record_id
                        + "\"" + "}");
        DataFields fc = new DataFields(snapshotPojoRow.fields.map(c -> c.toNamedDataField()));
        RecordChangeType recordChangeType = RecordChangeType.R.valueOf(data.method);
        RecordChange rc = new RecordChange(recordChangeType, COLLECTION, data.getRecord_id(), fc.toPutFieldChanges());
        return new Delta(rc);
    }

    @lombok.Data
    @ToString(of = {"uid", "record_rev", "record_id", "method"})
    @Bendable
    @AllArgsConstructor
    @NoArgsConstructor
    public static class Data implements Comparable<Data> {
        @BenderPart(name = "a.method", strictName = true)
        public String method;
        @BenderPart(name = "a.record_data", strictName = true)
        public Option<String> record_data;
        @BenderPart(name = "a.record_id", strictName = true)
        public String record_id;
        @BenderPart(name = "a.record_rev", strictName = true)
        public Integer record_rev;
        @BenderPart(name = "a.uid", strictName = true)
        public String uid;
        @BenderPart(name = "a.unixtime", strictName = true)
        public Long unixtime;

        @Override
        public int compareTo(@NotNull Data o) {
            return record_rev.compareTo(o.record_rev);
        }

        public void setRecordData(String data) {
            record_data = Option.of(data);
        }
    }

    @lombok.Data
    public static class DataCompacted {
        public DataApiUserId userId;
        public MapF<String, ListF<Delta>> deltas = Cf.linkedHashMap();
        public ListF<Delta> favoriteFolder = Cf.arrayList(genDelta(new Data(
                "insert",
                Option.of("[{\"field_id\":\"children\",\"value\":{\"type\":\"list\",\"list\":[]}},{\"field_id\":\"title\",\"value\":{\"type\":\"string\",\"string\":\"Favorites\"}}]"),
                "the_favorites_folder",
                2001,
                null,
                Instant.now().getMillis())));
        public ListF<Delta> rootFolder = Cf.arrayList(genDelta(new Data(
                "insert",
                Option.of("[{\"field_id\":\"children\",\"value\":{\"type\":\"list\",\"list\":[{\"type\":\"string\",\"string\":\"the_favorites_folder\"}]}},{\"field_id\":\"title\",\"value\":{\"type\":\"string\",\"string\":\"\"}}]"),
                "the_root_folder",
                2000,
                null,
                Instant.now().getMillis())));
    }
}
