#!/bin/sh -e

ec() {
    echo "$@" >&2
    "$@"
}

print_usage_and_exit() {
    echo "Usage: $0 (build|publish-new-version|build-deliver|deploy-and-balance|undeploy) (testing|production)" >&2
    exit 1
}

if [ "$#" -lt 1 ]; then
    print_usage_and_exit
fi

case $2 in
    "production")
        COKE_DEPLOY_ENVS=( elliptics-storage storage-srw )
        COKE_BALANCE_ENVS=( production )
        ;;

    "testing")
        COKE_DEPLOY_ENVS=( elliptics-test-storage storage-test-srw )
        COKE_BALANCE_ENVS=( testing )
        ;;

    *)
        if [ "$1" != "build" ] && [ "$1" != "publish-new-version" ]; then
            print_usage_and_exit
        fi
        ;;
esac

SCRIPT_DIR=$(cd -P -- "$(dirname -- "$0")" && pwd -P)
PRJ_DIR=${SCRIPT_DIR}/../../videostreaming
META_DIR=${PRJ_DIR}/src/main/cocaine
PKG_DIR=${PRJ_DIR}/cocaine_pkg
PROPS_DIR=${PKG_DIR}/props/ru/yandex/chemodan

third=$3

ec cd ${PRJ_DIR}

build() {
    ver=`python ${SCRIPT_DIR}/coke_version.py current --path=${META_DIR}`
    echo "##teamcity[buildNumber '$ver']"

    ec ../devel/ya make --checkout

    ec rm -rf ${PKG_DIR}
    ec mkdir ${PKG_DIR}

    ec cp ${META_DIR}/* ${PKG_DIR}

    ec mkdir -p ${PROPS_DIR}
    ec scp -r secdist.yandex.net:/repo/projects/dv/videostreaming-props/* ${PROPS_DIR}

    ec mkdir ${PKG_DIR}/lib
    ec cp ./disk-videostreaming/* ${PKG_DIR}/lib/
}

run_coke() {
    COKE_ENV=testing
    ec source ~/.venv/cocaine/bin/activate
    ec coke "$@"
}

inc_version() {
    curr_ver=`python ${SCRIPT_DIR}/coke_version.py current --path=${META_DIR}`
    next_ver=`python ${SCRIPT_DIR}/coke_version.py next_minor --path=${META_DIR}`
    ec perl -pi -e "s/$curr_ver/$next_ver/g" ${META_DIR}/project.json
    ec ../devel/ya svn commit -m "$next_ver SKIP_CHECK" ${META_DIR}/project.json
}

push() {
    ec cd ${PKG_DIR}
    run_coke push
}

deploy() {
    ec cd ${META_DIR}

    for env in "${COKE_DEPLOY_ENVS[@]}"
    do
        run_coke deploy ${env}
    done
}

balance() {
    ec cd ${META_DIR}

    for env in "${COKE_DEPLOY_ENVS[@]}"
    do
        run_coke balance ${env}
    done

    for env in "${COKE_BALANCE_ENVS[@]}"
    do
        run_coke balance ${env}
    done
}

deploy_and_balance() {
    deploy
    balance
}

undeploy() {
    if [ -z "$third" ]; then
        echo 'Version to undeploy not specified'
        print_usage_and_exit
    fi

    ec cd ${META_DIR}
    for env in "${COKE_DEPLOY_ENVS[@]}"
    do
        run_coke --no-config --set "version" "$third" --set "name" "disk-streaming" deploy -u ${env}
    done
}

case $1 in
    "build")
        build
        ;;

    "publish-new-version")
        inc_version
        build
        push
        ;;

    "deploy")
        deploy
        ;;

    "balance")
        balance
        ;;

    "build-deliver")
        build
        push
        deploy_and_balance
        ;;

    "deploy-and-balance")
        deploy_and_balance
        ;;

    "undeploy")
        undeploy
        ;;

    *)
        print_usage_and_exit
        ;;
esac
