#!/usr/bin/python
# -*- coding: utf-8 -*-

import os
import sys

from common import *
from configurator import Configurator
from gen_logrotate import gen_logrotate_conf
from gen_monrun import gen_monrun_conf
from gen_nginx import gen_nginx_conf
from gen_package_json import gen_package_json
from gen_ubic import gen_ubic_config, write_application_dockerized
from gen_version import gen_version_properties

qloud_testing_by_app = {
    "djfs-api": ["disk.disk-djfs-api.testing-stable"],
    "djfs-worker": [
        "disk.disk-djfs-worker.testing-stable",
        "disk.disk-djfs-worker.testing-hotfix"
    ],
    "countersapi": ["disk.disk-counters.testing"],
    "queller": [
        "disk.disk-queller.testing-stable",
        "disk.disk-queller.testing-current",
        "disk.disk-queller.testing-manual",
        "disk.disk-queller.testing-hotfix"
    ],
    "telemost-backend": [
        "disk.telemost-backend.testing"
    ],
    "docviewer": [
        "disk.disk-dv.testing",
        "disk.disk-dv.corp-testing"
    ]
}

qloud_prestable_by_app = {
    "monops": ["disk.monops.prestable"],
    "djfs-api": ["disk.disk-djfs-api.prestable-autotests"],
    "djfs-worker": ["disk.disk-djfs-worker.prestable-autotests"],
    "countersapi": ["disk.disk-counters.prestable"],
    "queller": [
        "disk.disk-queller.prestable-autotests",
        "disk.disk-queller.prestable-current",
        "disk.disk-queller.prestable-hotfix",
        "disk.disk-queller.prestable-iva"
    ],
    "docviewer": [
        "disk.disk-dv.prestable",
        "disk.disk-dv.corp-prestable"
    ]
}

def get_qloud_testing_object_ids(app):
    return qloud_testing_by_app.get(app, ["disk.disk-" + app + ".testing"])


def get_qloud_prestable_object_ids(app):
    return qloud_prestable_by_app.get(app, ["disk.disk-" + app + ".prestable"])


def _output_path(conf, path):
    return conf.eval_str(conf.base_path + "/.build_tmp" + path)


def prepare_package(conf, version_suffix=''):

    def eval_path(path):
        return conf.eval_str(path)

    def output_path(path):
        return _output_path(conf, path)

    def prepare_uploader():
        mkdir_p(output_path("/u0/${service}/${app}/tmp"))
        mkdir_p(output_path("/u0/${service}/${app}/local-files"))
        mkdir_p(output_path("/u0/${service}/${app}/queue/done"))
        mkdir_p(output_path("/u0/${service}/${app}/queue/failed"))
        for i in xrange(0, 256):
            mkdir_p(output_path("/u0/${service}/${app}/queue/processing/" + "%03d" % i))

        mkdir_p(output_path("/etc/yandex/${service}/${app}/keys"))
        for env in "development", "testing", "production":
            copy_safe("devel/etc/crossdomain/crossdomain-" + env + ".xml",
                      output_path("/etc/yandex/${service}/${app}/crossdomain/crossdomain-" + env + ".xml"))

        mkdir_p(output_path("/usr/share/doc/yandex-${service}-${app}/conf"))
        copy_safe("devel/logo.png", output_path("/usr/share/yandex-${service}-${app}/logo.png"))

        copy_safe("devel/bin/kqueue.sh", output_path("/usr/bin/kladun-kqueue.sh"))
        chmod_x(output_path("/usr/bin/kladun-kqueue.sh"))
        copy_safe("devel/bin/detect-active-network-interface.sh",
                  output_path("/usr/bin/detect-active-network-interface.sh"))
        chmod_x(output_path("/usr/bin/detect-active-network-interface.sh"))

    base_path = conf.base_path

    rm_safe(output_path(""))

    #base folders

    mkdir_p(output_path(""))

    mkdir_p(output_path("/usr/bin/yandex/${service}"))
    mkdir_p(output_path("/var/lib/yandex/${service}"))
    mkdir_p(output_path("/var/run/yandex/${service}"))
    mkdir_p(output_path("/var/tmp/yandex/${service}/${app}"))
    mkdir_p(output_path("/var/spool/yandex/${service}/${app}/tmp"))
    mkdir_p(output_path("/var/log/yandex/${service}/"))

    # specific preparations
    if conf.app == 'uploader':
        prepare_uploader()
    if conf.app == 'docviewer':
        copy_safe("devel/bin/connect.so", output_path("/usr/bin/connect.so"))
    # package json

    gen_package_json(base_path, output_path("/final_package.json"), version_suffix=version_suffix)

    # postinst
    mkdir_p(output_path("/debian"))
    conf.eval_file("debian/postinst2.template", output_path("/debian/postinst"))

    if os.path.exists(base_path + "/src/main/etc/"):
        copy_safe(base_path + "/src/main/etc/", output_path("/etc/"))

    # init.d

    copy_safe("debian/init-d.template", output_path("/etc/init.d/yandex-disk-${app}"))
    chmod_x(output_path("/etc/init.d/yandex-disk-${app}"))

    # java-oom-handler.sh

    copy_safe("devel/bin/java-oom-handler.sh",
              output_path("/usr/lib/yandex/${service}/${app}/bin/include/java-oom-handler.sh"))
    chmod_x(output_path("/usr/lib/yandex/${service}/${app}/bin/include/java-oom-handler.sh"))

    #ubic

    mkdir_p(output_path("/etc/ubic/service"))
    gen_ubic_config(base_path, output_path("/etc/ubic/service/" + conf.get_package_name() + ".ini"))
    # can't write json config - it breaks ubic
    # write_ubic_json_config(base_path, output_path("/etc/ubic/service/" + conf.get_package_name() + ".ini-json"))

    #ubic application_dockerized
    mkdir_p(output_path("/etc/yandex"))
    write_application_dockerized(base_path, output_path("/etc/yandex/"), "application-dockerized", False)
    write_application_dockerized(base_path, output_path("/etc/yandex/"), "java-exec", True)

    #version
    mkdir_p(output_path("/usr/lib/yandex/${service}/${app}/lib"))
    gen_version_properties(output_path("/usr/lib/yandex/${service}/${app}/lib/application-version.properties"))

    #monrun

    mkdir_p(output_path("/etc/monrun/conf.d/"))
    gen_monrun_conf(base_path, output_path("/etc/monrun/conf.d/" + conf.get_app() + ".conf"))

    #nginx
    mkdir_p(output_path("/etc/nginx/sites-enabled"))
    gen_nginx_conf(base_path, output_path("/etc/nginx/sites-enabled"))
    if os.path.exists(base_path + "/src/main/etc/nginx/sites-enabled"):
        conf.eval_folder(base_path + "/src/main/etc/nginx/sites-enabled", output_path("/etc/nginx/sites-enabled"))

    #logtotate
    #todo: check and remove this copy
    if os.path.exists(base_path + "/src/main/etc/logrotate.d"):
        copy_safe(base_path + "/src/main/etc/logrotate.d", output_path("/etc/logrotate.d"))
    else:
        mkdir_p(output_path("/etc/logrotate.d"))
        gen_logrotate_conf(base_path, output_path("/etc/logrotate.d/yandex-${service}-${app}.conf"))

    #properties

    properties_path = output_path("/etc/yandex/${service}/${app}")
    mkdir_p(properties_path)

    support_props_base = "support/common/src/main/java/ru/yandex/chemodan/global"
    copy_safe(support_props_base + "/application-default.properties", properties_path + "/global-application-default.properties")
    copy_safe(support_props_base + "/application-development.properties", properties_path + "/global-application-development.properties")
    copy_safe(support_props_base + "/application-testing.properties", properties_path + "/global-application-testing.properties")
    copy_safe(support_props_base + "/application-stress.properties", properties_path + "/global-application-stress.properties")
    copy_safe(support_props_base + "/application-prestable.properties", properties_path + "/global-application-prestable.properties")
    copy_safe(support_props_base + "/application-production.properties", properties_path + "/global-application-production.properties")

    # copy to /etc/yandex/ for common usage
    copy_safe(support_props_base + "/jaas-testing.conf", output_path("/etc/yandex/jaas-testing.conf"))

    component_props_base = base_path + "/src/main/java/ru/yandex/chemodan/"
    copy_safe(eval_path(component_props_base + "/app-${app}-default.properties"), properties_path + "/application-default.properties")
    copy_safe(eval_path(component_props_base + "/app-${app}-development.properties"), properties_path + "/application-development.properties")
    copy_safe(eval_path(component_props_base + "/app-${app}-testing.properties"), properties_path + "/application-testing.properties")
    copy_safe(eval_path(component_props_base + "/app-${app}-stress.properties"), properties_path + "/application-stress.properties")
    copy_safe(eval_path(component_props_base + "/app-${app}-prestable.properties"), properties_path + "/application-prestable.properties")
    copy_safe(eval_path(component_props_base + "/app-${app}-production.properties"), properties_path + "/application-production.properties")


def build_package(base_path, raw_args):
    conf = Configurator(base_path)
    args_dict = get_arg_dict(raw_args)

    build_tar = "--tar" in args_dict
    build_deb = "--debian" in args_dict
    upload = "--upload" in args_dict
    deploy_testing = "--deploy-testing" in args_dict
    deploy_prestable = "--deploy-prestable" in args_dict

    args = [k + '=' + v if v is not None else k for k, v in args_dict.items() if k not in ["--tar", "--debian", "--upload", "--deploy-testing", "--deploy-prestable", "--version-suffix"]]

    prepare_package(conf, version_suffix=args_dict.get('--version-suffix', ''))
    package_json = _output_path(conf, "/final_package.json")

    if build_tar:
        ya_package(args + ["--tar"], package_json)

        if upload:
            print "going to upload tar to sandbox"

            disk_root_dir = os.path.dirname(os.path.dirname((os.path.dirname(os.path.abspath(__file__)))))

            for path in os.listdir(disk_root_dir):
                # find all files, ending with '-default.properties'
                if os.path.isfile(path) and path.find('.tar.gz') >= 0:
                    package_version = path[path.find(".") + 1:path.find(".tar.gz")]

                    print "##teamcity[buildNumber '" + package_version + "']"

                    ya(['upload', '--ttl=inf', '-T=DISK_COMPRESSED_RESOURCE_APPLICATION',
                        '-d="' + conf.get_package_name() + '=' + package_version + '"',
                        '-a=linux', '--owner=DISK-ADMIN',
                        '-A ' + conf.get_package_name() + '=' + package_version,
                         path])

                    if deploy_testing or deploy_prestable:
                        ya(["make", "admin/qloud-deploy", "--checkout"])

                    if deploy_testing:
                        for obj_id in get_qloud_testing_object_ids(conf.get_app()):
                            qloud_deploy(obj_id, conf.get_package_name() + '=' + package_version)

                    if deploy_prestable:
                        for obj_id in get_qloud_prestable_object_ids(conf.get_app()):
                            qloud_deploy(obj_id, conf.get_package_name() + '=' + package_version)

            os.system("rm *.tar.gz")

    if build_deb:
        deb_args = args + ["--debian"]
        if upload:
            deb_args += ["--publish-to=yandex-disk-common"]
            if 'GPG_SECRET_KEY' in os.environ:
                deb_args += ["--key=robot-disk-j-build@yandex-team.ru"]

        ya_package(deb_args, package_json)
        os.system("rm *.tar.gz")

    rm_safe(_output_path(conf, ""))


def get_arg_dict(args):
    return {kv[0]: kv[1] if len(kv) >= 2 else None for kv in [a.split('=', 1) for a in args]}


if __name__ == "__main__":
    # chdir to arcadia/disk
    os.chdir(os.path.abspath(os.path.join(os.path.abspath(sys.argv[0]), os.pardir, os.pardir, os.pardir)))

    if sys.argv[-1] == "--dry-run":
        prepare_package(Configurator(sys.argv[1]), version_suffix=get_arg_dict(sys.argv[2:]).get('--version-suffix', ''))
    else:
        build_package(sys.argv[1], sys.argv[2:])
