#!/usr/bin/python
# -*- coding: utf-8 -*-

import collections
import json
import os.path
import re

from collections import OrderedDict

def _read_properties(path):
    properties = dict()
    not_finished = None

    for line in open(path, "r").readlines():
        if not_finished is not None:
            value = line.strip()
            if len(value) > 0 and value[-1] == "\\":
                value = value[:-1]
            else:
                not_finished = None

            properties[key] = properties[key] + value

        elif line.find("=") > 0:
            i = line.find("=")
            key = line[:i].strip()
            value = line[i + 1:].strip()

            if len(value) > 0 and value[-1] == "\\":
                value = value[:-1]
                not_finished = key

            properties[key] = value
    return properties


class Configurator:
    def __init__(self, base_path):
        while base_path[-1] == "/":
            base_path = base_path[:-1]

        self.base_path = base_path
        self.properties = self._load_properties()

        self.app = self.properties["app"]
        self.service = self.properties["service"]

    def _load_properties(self):
        properties = {
            "component_root": self.base_path,
            "current_version": self.get_current_version()
        }

        component_properties_path = self.base_path + '/src/main/resources/ru/yandex/chemodan/'
        if not os.path.exists(component_properties_path):
            component_properties_path = self.base_path + '/src/main/java/ru/yandex/chemodan/'

        properties.update(_read_properties("support/ubic-default.properties"))

        common_properties = _read_properties("support/common/src/main/java/ru/yandex/chemodan/global/application-default.properties")
        properties.update(common_properties)

        for f in os.listdir(component_properties_path):
            path = os.path.join(component_properties_path, f)
            # find all files, ending with '-default.properties'
            if os.path.isfile(path) and f.find('default.properties') >= 0:
                component_properties = _read_properties(path)
                properties.update(component_properties)

        if not "app" in properties:
            main_class = properties["main.class"]
            pattert = re.compile(".*return new SimpleAppName\( *\"([^\"]*)\" *, *\"([^\"]*)\" *\).*")
            for line in open(self.base_path + "/src/main/java/" + main_class.replace(".", "/") + ".java", "r").readlines():
                res = pattert.match(line)
                if res is not None:
                    properties["app"] = res.group(2)
                    if "service" not in properties:
                        properties["service"] = res.group(1)
                    break

        app = properties["app"]
        service_prefix_common = "service_" + properties["service"] + "_all"

        more_component_properties = _read_properties(component_properties_path + 'app-' + app + '-default.properties')
        properties.update(more_component_properties)

        cut_add_name = {}
        cut_app_name = {}
        cut_service_name = {}
        service_prefix = service_prefix_common + "."

        for k, v in properties.iteritems():
            if k.find(app + ".") == 0:
                cut_app_name[k[len(app) + 1:]] = v
                continue
            if k.find(service_prefix) == 0:
                cut_service_name[k[len(service_prefix):]] = v

        cut_add_name.update(cut_app_name)
        for k, v in cut_service_name.iteritems():
            if k not in cut_app_name:
                cut_add_name[k] = v

        properties.update(cut_add_name)

        if app == "docviewer":
            properties["app-prefix"] = service_prefix_common
        else:
            properties["app-prefix"] = app

        return properties

    def get_app(self):
        return self.app

    def get_service(self):
        return self.service

    def get_package_name(self):
        return "yandex-disk-" + self.get_app()

    def eval_str(self, property_str):
        while property_str.find("${") >= 0:
            i1 = property_str.rfind("${")
            if i1 < 0:
                return property_str

            i2 = property_str.find("}", i1)
            if i2 < 0:
                raise "Unterminated ${"

            property_name = property_str[i1 + 2:i2]
            property_str = property_str[0:i1] + self.eval_str(self.properties[property_name]) + property_str[i2 + 1:]

        return property_str

    def load_json(self, *paths):
        result = {}
        for path in paths:
            if os.path.isfile(path):
                content = "".join(open(path, "r").readlines())
                self._deep_dict_merge(result, json.loads(content, object_pairs_hook=OrderedDict))

        result = self._eval_json(result)

        return result

    def _eval_json(self, js):
        if isinstance(js, (str, unicode)):
            return self.eval_str(js)

        if isinstance(js, dict):
            result = OrderedDict()
            for k,v in js.iteritems():
                result[k] = self._eval_json(v)
            return result

        elif isinstance(js, list):
            result = []
            for elem in js:
                result.append(self._eval_json(elem))

            return result

        return js

    def _deep_dict_merge(self, result, merge_dict):
        for k, v in merge_dict.iteritems():
            if k in result and isinstance(result[k], dict) and isinstance(merge_dict[k], collections.Mapping):
                self._deep_dict_merge(result[k], merge_dict[k])
            else:
                result[k] = merge_dict[k]

    def get_current_version(self):
        line = open("debian/changelog").readline()
        match = re.search("^.*\(([^\.]*\.[^\.]*..*)\).*$", line)
        if not match:
            print "Can't find current version in changelog"
            sys.exit(1)
        return match.group(1)

    def eval_file(self, src, dst):
        out = open(dst, "w")
        for line in open(src, "r").readlines():
            out.write(self.eval_str(line))
        out.close()

    def eval_folder(self, src_path, output_dir):
        for file in os.listdir(src_path):
            self.eval_file(os.path.join(src_path, file), os.path.join(output_dir, self.eval_str(file)))
