#!/usr/bin/python
# -*- coding: utf-8 -*-

import json
import sys

from common import *
from configurator import Configurator

TAB = " " * 4
TABx2 = TAB * 2
PARAM_SEP = " \\\n"
UBIC_INFO_EXCLUDES = {"java_args", "java15_path", "java17_path"}

def write_ubic_info(ubic_info, output_path):
    out = open(output_path, "w")
    out.write("module = Media::JavaUbic\n\n[options]\n\n")

    for k,v in ubic_info.iteritems():
        if k not in UBIC_INFO_EXCLUDES:
            out.write(k)
            out.write(" = ")
            out.write(str(v))
            out.write("\n")

    print_ubic_java_args(out, ubic_info.get("java_args", {}))

    out.flush()
    out.close()


def print_ubic_java_args(out, java_args):
    print_ubic_one_java_args(out, java_args.get("default", {}), "")
    out.write("\n")

    print_ubic_one_java_args(out, java_args.get("development", {}), "development")
    out.write("\n")

    print_ubic_one_java_args(out, java_args.get("testing", {}), "testing")
    out.write("\n")

    print_ubic_one_java_args(out, java_args.get("stress", {}), "stress")
    out.write("\n")

    print_ubic_one_java_args(out, java_args.get("prestable", {}), "prestable")
    out.write("\n")


def print_ubic_one_java_args(out, java_args, env):
    for k,v in java_args.iteritems():
        out.write("java_args_")
        out.write(k)
        if len(env) > 0:
            out.write("@" + env)
        out.write(" = " + v + "\n")


def get_ubic_json_config(conf):
    return conf.load_json("support/ubic_config.json", conf.base_path + "/ubic_config.json")


def write_ubic_json_config(base_path, output_path):
    ubic_json = get_ubic_json_config(Configurator(base_path))

    out = open(output_path, "w")
    out.write(json.dumps(ubic_json, indent=4, separators=(',', ': ')))
    out.close()


def write_env_application_dockerized(conf, environment, path, is_exec):
    ubic_config = get_ubic_json_config(conf)

    out = open(path, "w")

    out.write("#!")
    out.write(ubic_config['shell'])
    out.write("\n\n")

    out.write('if [ "$USE_JDK15" == "yes" ]\n')
    out.write('then\n')
    write_java_exec(conf, environment, is_exec, out, ubic_config, 'java15_path')
    out.write('elif [ "$USE_JDK17" == "yes" ]\n')
    out.write('then\n')
    write_java_exec(conf, environment, is_exec, out, ubic_config, 'java17_path')
    out.write('else\n')
    write_java_exec(conf, environment, is_exec, out, ubic_config, 'java_path')
    out.write('fi\n')

    out.close()
    chmod_x(path)


def write_java_exec(conf, environment, is_exec, out, ubic_config, java_path_key):
    args = ubic_config['java_args']['default'].copy()
    args.update(ubic_config['java_args'].get(environment, {}))

    out.write(TAB)
    if is_exec:
        out.write("exec ")

    out.write(ubic_config.get(java_path_key + '@' + environment, ubic_config[java_path_key]))
    out.write(PARAM_SEP)

    out.write(PARAM_SEP.join([TABx2 + a for a in args.values() if a]))
    out.write(PARAM_SEP)

    if ubic_config['exit_on_oom'] == "1":
        out.write(TABx2)
        out.write(conf.eval_str("-XX:OnOutOfMemoryError='/usr/lib/yandex/${service}/${app}/bin/include/java-oom-handler.sh -p %p &'"))
        out.write(PARAM_SEP)

    out.write(TABx2)
    out.write("-classpath '" + ubic_config['classpath'] + "'")
    out.write(PARAM_SEP)

    out.write(TABx2)
    out.write(ubic_config['main_class'])
    out.write('\n')


def write_application_dockerized(base_path, folder_path, name, is_exec):
    conf = Configurator(base_path)
    write_env_application_dockerized(conf, "default", folder_path + "/" + name + ".default", is_exec)
    write_env_application_dockerized(conf, "development", folder_path + "/" + name + ".development", is_exec)
    write_env_application_dockerized(conf, "testing", folder_path + "/" + name + ".testing", is_exec)
    write_env_application_dockerized(conf, "stress", folder_path + "/" + name + ".stress", is_exec)
    write_env_application_dockerized(conf, "prestable", folder_path + "/" + name + ".prestable", is_exec)
    write_env_application_dockerized(conf, "stable", folder_path + "/" + name + ".stable", is_exec)


def gen_ubic_config(base_path, output_path):
    ubic_info = get_ubic_json_config(Configurator(base_path))

    write_ubic_info(ubic_info, output_path)


if __name__ == "__main__":
    base_path = sys.argv[1]

    gen_ubic_config(base_path, base_path + "/src/main/etc/ubic/service/" + conf.get_package_name() + ".ini")
