#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys

from time import time, mktime, strptime

ARGS_FORMAT = "%Y.%m.%dT%H:%M:%S"

LOG_FORMAT = "%Y-%m-%d %H:%M:%S"

TEMPLATE = '{count} tasks sent to {destination} in {duration}s. rps = {rps}'


def parse_time(dts):
    try:
        return mktime(strptime(dts, ARGS_FORMAT))
    except ValueError:
        return None


def parse_time_limits(dts_list):
    limits = [parse_time(dts) for dts in dts_list + ['', '']][:2]
    return [given or default for given, default in zip(limits, (0, time()))]


def parse_log_line(line):
    result = {'timestamp': -1, 'celery_inc': 0, 'bazinga_inc': 0}
    data = dict([element.split('=', 1) for element in line.split('\t')[1:]])
    if not data:
        return result

    message = data['message']
    result['timestamp'] = mktime(strptime(data['timestamp'], LOG_FORMAT))
    if 'saved to be executed' in message:
        result['bazinga_inc'] = 1
    elif 'confirmed' in message:
        submsg = message[20:]
        submsg = submsg[:submsg.find(' messages')]
        result['celery_inc'] = float(submsg)

    return result


def update(struct, key, timestamp, inc):
    if inc > 0:
        if key not in struct:
            struct[key] = {'start': timestamp,
                           'stop': timestamp,
                           'value': inc}
        else:
            struct[key]['stop'] = timestamp
            struct[key]['value'] += inc


if __name__ == '__main__':
    logfile = sys.argv[1]
    start, stop = parse_time_limits(sys.argv[2:4])

    result = {}

    with open(logfile, 'r') as f:
        for line in f:
            data = parse_log_line(line)
            timestamp = data['timestamp']
            if start < timestamp < stop:
                update(result, 'celery', timestamp, data['celery_inc'])
                update(result, 'bazinga', timestamp, data['bazinga_inc'])

    for key, data in result.items():
        duration = data['stop'] - data['start']
        print TEMPLATE.format(count=data['value'],
                              destination=key,
                              duration=duration,
                              rps=data['value'] / (duration or 1))
