package ru.yandex.chemodan.app.djfs.migrator;

import java.util.UUID;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.pg.SharpeiShardResolver;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.operations.OperationDao;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.app.djfs.migrator.migrations.DjfsMigrationPlan;
import ru.yandex.chemodan.app.djfs.migrator.migrations.DjfsTableMigration;
import ru.yandex.chemodan.util.sharpei.SharpeiClient;
import ru.yandex.chemodan.util.sharpei.SimpleUserId;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.test.Assert;

/**
 * @author yappo
 */
@RequiredArgsConstructor
public class DjfsMigrator {
    public static final Duration RESENT_OPERATIONS_FAIL_DURATION = Duration.standardHours(1);

    private static final ListF<Operation.State> WRITE_OPERATIONS =
            Cf.list(Operation.State.WAITING, Operation.State.EXECUTING);
    private static final ListF<String> NOT_CHECK_OPERATION_TYPES = Cf.list("download", "social");

    private static final Logger logger = LoggerFactory.getLogger(DjfsMigrator.class);

    private final DjfsMigratorConfigFactory djfsMigratorConfigFactory;
    private final LockManager lockManager;
    private final SharpeiClient sharpeiClient;
    private final SharpeiShardResolver sharpeiShardResolver;
    private final OperationDao operationDao;

    public CopyResult copyDataAndSwitchShard(DjfsUid uid, int fromShardId, int destinationShardId, boolean forceActive,
            DjfsMigrationPlan migrationPlan)
    {
        Assert.notEquals(fromShardId, destinationShardId, "src shard and dst shard must be different");
        if (alreadyMigrated(uid, destinationShardId)) {
            return new CopyResult(DjfsMigrationState.ALREADY_MIGRATED, "user already migrated");
        }
        Assert.equals(currentUserShardId(uid), fromShardId, "current user shard and srcShard must be same");
        if (lockManager.isLocked(uid)) {
            return new CopyResult(DjfsMigrationState.USER_ALREADY_LOCKED_FOR_MIGRATION,
                    "user already in migration process");
        }
        if (hasResentOperations(uid, forceActive)) {
            return new CopyResult(DjfsMigrationState.USER_IS_ACTIVE, "user has recent active operations");
        }

        DjfsCopyConfiguration migrationConf = djfsMigratorConfigFactory
                .copyConfigBuilder(uid, fromShardId, destinationShardId)
                .build();

        PgSchema sourceSchema = PgSchema.build(migrationConf.srcShardJdbcTemplate());
        try {
            PgSchema destinationSchema = PgSchema.build(migrationConf.dstShardJdbcTemplate());
            checkSchemaIsSame(sourceSchema, destinationSchema);
            checkAllTablesInMigrationPlan(sourceSchema, migrationPlan);
        } catch (AssertionError e) {
            return new CopyResult(DjfsMigrationState.MIGRATION_CONFIG_ERROR, e.getMessage());
        }

        try {
            MigrationLock migrationLock = new MigrationLock(
                    lockManager, uid, migrationConf.getSrcShardInfo(), migrationConf.getDstShardInfo()
            );
            return migrationLock.withLockOnDst(() -> {
                        cleanData(migrationConf.getUid(), migrationPlan, migrationConf.dstShardJdbcTemplate(),
                                migrationConf.getBaseBatchSize());
                        return copyDataInternal(migrationPlan, migrationConf, sourceSchema, forceActive, migrationLock);
                    }
            );
        } catch (Exception e) {
            logger.error("error on copying {} from {} to {}", uid, fromShardId, destinationShardId, e);
            return new CopyResult(DjfsMigrationState.EXCEPTION_ON_COPYING, e.getMessage());
        }
    }

    private boolean alreadyMigrated(DjfsUid uid, int destinationShardId) {
        return currentUserShardId(uid) == destinationShardId;
    }

    @NotNull
    private CopyResult copyDataInternal(DjfsMigrationPlan migrationPlan, DjfsCopyConfiguration migrationConf,
            PgSchema pgSchema, boolean forceActive, MigrationLock migrationLock)
    {
        DjfsUid uid = migrationConf.getUid();

        if (hasActiveOperations(uid, forceActive)) {
            return new CopyResult(DjfsMigrationState.USER_IS_ACTIVE, "user has active operations");
        }
        return measureTime(migrationConf.srcShardJdbcTemplate(), migrationConf.getUid(), "Migration lock",
                () -> migrationLock.withLockOnSrc(() -> {
                    //double check for case of inserted operation between check and getting lock
                    if (hasActiveOperations(uid, forceActive)) {
                        return new CopyResult(DjfsMigrationState.USER_IS_ACTIVE, "user has active operations");
                    }
                    copyData(migrationPlan, migrationConf, migrationLock, pgSchema);

                    checkAllCopied(migrationPlan, migrationConf, pgSchema);

                    RetryUtils.retry(3, 50, 1.0, () -> switchShard(uid, migrationConf.getDstShardInfo().getShardId()));
                    return new CopyResult(DjfsMigrationState.COPY_SUCCESS, "success migration");
                })
        );
    }

    private CopyResult measureTime(JdbcTemplate3 shard, DjfsUid uid, String message, Function0<CopyResult> block) {
        long filesCount = shard.queryForLong("SELECT count(*) FROM disk.files WHERE uid = ?", uid);
        long foldersCount = shard.queryForLong("SELECT count(*) FROM disk.folders WHERE uid = ?", uid);
        Instant begin = Instant.now();
        CopyResult result = block.apply();
        Duration takeTime = new Duration(begin, Instant.now());
        logger.info("{} for uid {} take {} ({} ms per file, {} ms per file+folder) files count {} folders count {}",
                message, uid, takeTime,
                msPerCount(takeTime, filesCount),
                msPerCount(takeTime, filesCount + foldersCount),
                filesCount, foldersCount
        );
        return result;
    }

    private double msPerCount(Duration takeTime, double filesCount) {
        return Math.round(takeTime.getMillis() * 1000 / filesCount) / 1000.0;
    }

    public void cleanData(DjfsUid uid, int onShardId, ListF<Integer> cleanLocksOn, DjfsMigrationPlan migrationPlan) {
        Assert.notEquals(currentUserShardId(uid), onShardId, "current user shard and shard to clean must be different");

        DjfsCleanConfiguration cleanConfiguration =
                djfsMigratorConfigFactory.cleanConfigBuilder(uid, onShardId).build();
        PgSchema pgSchema = PgSchema.build(cleanConfiguration.shardJdbcTemplate());
        checkAllTablesInMigrationPlan(pgSchema, migrationPlan);
        cleanData(
                cleanConfiguration.getUid(),
                migrationPlan,
                cleanConfiguration.shardJdbcTemplate(),
                cleanConfiguration.getBaseBatchSize()
        );
        for (Integer shardIdToCleanLock : cleanLocksOn) {
            lockManager.cleanUpMigrationLock(uid, new DjfsShardInfo.Pg(shardIdToCleanLock));
        }
    }

    private int currentUserShardId(DjfsUid uid) {
        return sharpeiShardResolver.shardByUid(uid)
                .getOrThrow(() -> new UserNotInitializedException(uid, "no sharpei entry")).getShardId();
    }

    private void cleanData(DjfsUid uid, DjfsMigrationPlan migrationPlan, JdbcTemplate3 onShard, int batchSize) {
        for (DjfsTableMigration migration : migrationPlan.getMigrations().reverse()) {
            logger.info("cleaning data of {} {}", migration.getClass().getSimpleName(), migration.tables());
            migration.cleanData(onShard, uid, batchSize);
        }
    }

    private void copyData(DjfsMigrationPlan migrationPlan, DjfsCopyConfiguration migrationConf,
            MigrationLock migrationLock, PgSchema sourceSchema)
    {
        for (DjfsTableMigration migration : migrationPlan.getMigrations()) {
            logger.info("copying data of {} {}", migration.getClass().getSimpleName(), migration.tables());
            migration.runCopying(migrationConf, sourceSchema, migrationLock::updateLock);
        }
    }

    private void checkAllCopied(DjfsMigrationPlan migrationPlan, DjfsCopyConfiguration migrationConf,
            PgSchema pgSchema)
    {
        for (DjfsTableMigration migration : migrationPlan.getMigrations()) {
            logger.info("checking copied data of {} {}", migration.getClass().getSimpleName(), migration.tables());
            migration.checkAllCopied(migrationConf, pgSchema);
        }
    }

    private boolean hasActiveOperations(DjfsUid uid, boolean forceActive) {
        if (forceActive) {
            return false;
        }
        return operationDao.count(uid, WRITE_OPERATIONS, NOT_CHECK_OPERATION_TYPES) > 0;
    }

    private boolean hasResentOperations(DjfsUid uid, boolean forceActive) {
        if (forceActive) {
            return false;
        }
        return operationDao.count(uid, Instant.now().minus(RESENT_OPERATIONS_FAIL_DURATION)) > 0;
    }

    private void switchShard(DjfsUid uid, int destinationShardId) {
        int srcShardId = sharpeiShardResolver.shardByUid(uid)
                .getOrThrow(() -> new UserNotInitializedException(uid, "no sharpei entry")).getShardId();
        sharpeiClient.updateUser(
                new SimpleUserId(uid.asString()),
                Option.of(Tuple2.tuple(srcShardId, destinationShardId)),
                Option.empty()
        );
    }

    private static void checkAllTablesInMigrationPlan(PgSchema sourceSchema, DjfsMigrationPlan migrationPlan) {
        SetF<String> tablesInDb = sourceSchema.getTables().keySet();

        SetF<String> tablesWithMigrations = migrationPlan.getTablesOrder().unique();
        Assert.isEmpty(tablesWithMigrations.minus(tablesInDb), "some tables to migrate not in database");

        SetF<String> knownTables = tablesWithMigrations.plus(migrationPlan.getIgnore());
        Assert.isEmpty(tablesInDb.minus(knownTables), "some tables in database not mentioned in migrationPlan");
    }

    private void checkSchemaIsSame(PgSchema sourceSchema, PgSchema destinationSchema) {
        Assert.equals(
                sourceSchema.getTables().keySet(),
                destinationSchema.getTables().keySet()
        );

        for (String tableName : sourceSchema.getTables().keys()) {
            compareTables(
                    sourceSchema.getTables().getTs(tableName),
                    destinationSchema.getTables().getTs(tableName)
            );
        }
    }

    private void compareTables(PgSchema.Table first, PgSchema.Table second) {
        Assert.equals(
                first.getColumns().map(PgSchema.Column::getName).unique(),
                second.getColumns().map(PgSchema.Column::getName).unique()
        );
        Assert.equals(
                first.getColumns().unique(),
                second.getColumns().unique()
        );
    }

    /**
     * Лочим пользователя как на обоих шардах. В случае ошибки синмаем лок на обоих.
     * <p>
     * В случае успешной миграции оставляем на пару часов лок на шарде, откуда копировали
     * на случай если кто-то попытается изменить уже скопированные данные.
     */
    @RequiredArgsConstructor(access = AccessLevel.PRIVATE)
    public static class MigrationLock {
        static final Duration UPDATE_LOCK_AFTER = Duration.standardMinutes(5);
        static final Duration LOCK_FOR = Duration.standardMinutes(10);
        public static final Duration LOCK_SOURCE_SHARD_AFTER_MIGRATION_FOR = Duration.standardHours(2);

        private final LockManager lockManager;
        private final DjfsUid uid;
        private final DjfsShardInfo.Pg src;
        private final DjfsShardInfo.Pg dst;
        private final String ownerMark = UUID.randomUUID().toString();
        private Option<Instant> lastLockCall;

        private CopyResult withLockOnSrc(Function0<CopyResult> block) {
            lockManager.lockForMigration(uid, src, LOCK_FOR, ownerMark);
            lastLockCall = Option.of(lastLockCall.getOrElse(Instant.now()));
            try {
                CopyResult result = block.apply();
                if (result.getState() == DjfsMigrationState.COPY_SUCCESS) {
                    lockManager.updateLockForMigration(uid, src, LOCK_SOURCE_SHARD_AFTER_MIGRATION_FOR, ownerMark);
                } else {
                    RetryUtils.retry(3, () -> lockManager.unlockForMigration(uid, src, ownerMark));
                }
                return result;
            } catch (Throwable e) {
                logger.error("error within withUserLock", e);
                RetryUtils.retry(3, () -> lockManager.unlockForMigration(uid, src, ownerMark));
                throw e;
            }
        }

        private CopyResult withLockOnDst(Function0<CopyResult> block) {
            lockManager.lockForMigration(uid, dst, LOCK_FOR, ownerMark);
            lastLockCall = Option.of(Instant.now());
            try {
                return block.apply();
            } finally {
                RetryUtils.retry(3, () -> lockManager.unlockForMigration(uid, dst, ownerMark));
            }
        }

        private void updateLock() {
            Assert.some(lastLockCall);
            if (lastLockCall.get().plus(UPDATE_LOCK_AFTER).isBefore(Instant.now())) {
                lockManager.updateLockForMigration(uid, src, LOCK_FOR, ownerMark);
                lockManager.updateLockForMigration(uid, dst, LOCK_FOR, ownerMark);
                lastLockCall = Option.of(Instant.now());
            }
        }
    }

    @RequiredArgsConstructor
    @Getter
    public static class CopyResult {
        private final DjfsMigrationState state;
        private final String message;
    }
}
