package ru.yandex.chemodan.app.djfs.migrator;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.ConstructorArgumentValues;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.web.ConnectionIdActionDispatcherInterceptor;
import ru.yandex.chemodan.app.djfs.core.web.DjfsTskvLog4jRequestLog;
import ru.yandex.chemodan.app.djfs.core.web.JsonStringResultSerializer;
import ru.yandex.chemodan.app.djfs.core.web.LoggingAnyExceptionHandler;
import ru.yandex.chemodan.app.djfs.migrator.tasks.DjfsMigratorCleanTask;
import ru.yandex.chemodan.app.djfs.migrator.tasks.DjfsMigratorCleanUpLocksTask;
import ru.yandex.chemodan.app.djfs.migrator.tasks.DjfsMigratorCopyTask;
import ru.yandex.chemodan.app.djfs.migrator.tasks.DjfsMigratorYtSupplyTask;
import ru.yandex.chemodan.bazinga.BazingaWorkerTaskQueues;
import ru.yandex.chemodan.bazinga.ChemodanBazingaWorkerContextConfiguration;
import ru.yandex.chemodan.util.sharpei.SharpeiClient;
import ru.yandex.chemodan.util.web.A3JettyConfiguration;
import ru.yandex.chemodan.util.web.A3JettyContextConfiguration;
import ru.yandex.chemodan.util.web.MasterSlaveUnavailableExceptionHandler;
import ru.yandex.chemodan.util.web.SuppressInvocationInfoJsonPojoResultSerializer;
import ru.yandex.chemodan.util.web.interceptors.ThreadLocalCacheInterceptor;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.a3.ActionApp;
import ru.yandex.commune.a3.ActionConfigurationBuilder;
import ru.yandex.commune.a3.ActionConfigurator;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.parameter.bind.BenderJsonListParameterBinder;
import ru.yandex.commune.a3.action.parameter.bind.BenderParameterBinder;
import ru.yandex.commune.a3.action.parameter.convert.DefaultConverters;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.commune.a3.action.result.ResponseInterceptor;
import ru.yandex.commune.a3.action.result.SimpleResultSerializer;
import ru.yandex.commune.a3.action.result.error.A3ExceptionHandler;
import ru.yandex.commune.a3.action.result.error.HttpStatusCodeSourceExceptionHandler;
import ru.yandex.commune.a3.action.result.error.nested.ExceptionWithRootElementHandler;
import ru.yandex.commune.a3.action.result.type.MediaType;
import ru.yandex.commune.a3.db.MasterSlavePolicyInterceptor;
import ru.yandex.commune.a3.security.SecurityExceptionHandler;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.TaskQueue;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.ReadableInstantConfigurableMarshaller;
import ru.yandex.misc.spring.ApplicationContextUtils;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;

/**
 * @author yappo
 */
@Import({
        A3JettyContextConfiguration.class,
        DjfsMigratorContext.class,
        ChemodanBazingaWorkerContextConfiguration.class
})
public class DjfsMigratorApplicationContext {

    @Configuration
    public static class YtConfig {
        @Bean
        public Yt dataApiYtClient(
                @Value("${djfs-migrator.yt.http.proxy}") String ytHttpProxyUrl,
                @Value("${djfs-migrator.yt.user.yt.token}") String ytUserOauthToken
        ) {
            return YtUtils.http(ytHttpProxyUrl, ytUserOauthToken);
        }

        @Bean
        public DjfsMigratorYtSupplier djfsMigratorYtSupplier(Yt ytClient, BazingaTaskManager bazingaTaskManager,
                BazingaStorage bazingaStorage,
                @Value("${djfs-migrator.yt-import.retries-count}") int retiresCount,
                @Value("${djfs-migrator.yt-import.retries-interval}") Duration retiresInterval
        ) {
            RetryPolicy retryPolicy = new RetryPolicy()
                    .withMaxRetries(retiresCount)
                    .withDelay(retiresInterval.getMillis(), TimeUnit.MILLISECONDS);
            YtHelper ytHelper = new YtHelper(ytClient, retryPolicy);
            return new DjfsMigratorYtSupplier(ytHelper, bazingaTaskManager, bazingaStorage);
        }
    }

    @Configuration
    public static class A3 {
        @Bean
        public A3JettyConfiguration a3JettyServletsConfiguration() {
            return new A3JettyConfiguration(Tuple2List.fromPairs("/api/*", "")) {
                @Override
                public void postConstruct(SingleWarJetty jetty, ActionApp actionApp) {
                    jetty.setRequestLogFactory(DjfsTskvLog4jRequestLog::new);
                }
            };
        }

        @Bean
        public ActionApp actionApp(ApplicationContext context, ApplicationInfo applicationInfo) {
            BenderMapper mapper = new BenderMapper(BenderConfiguration.cons(
                    MembersToBind.WITH_ANNOTATIONS, false,
                    CustomMarshallerUnmarshallerFactoryBuilder.cons()
                            .add(Instant.class, new ReadableInstantConfigurableMarshaller(ISODateTimeFormat.dateTime()))
                            .build()
            ));

            ListF<CloneableAction> cloneableActions = ApplicationContextUtils.beansOfType(context, CloneableAction.class);

            ListF<Object> actionContainers =
                    ApplicationContextUtils.beansWithAnnotationList(context, ActionContainer.class).get2();

            ActionConfigurationBuilder builder = ActionConfigurationBuilder
                    .cons(applicationInfo)
                    .setConverters(DefaultConverters.all().getConverters())
                    .setExceptionHandlers(Cf.list(
                            new MasterSlaveUnavailableExceptionHandler(),
                            new HttpStatusCodeSourceExceptionHandler(),
                            new ExceptionWithRootElementHandler(),
                            new SecurityExceptionHandler(),
                            new A3ExceptionHandler(),
                            new LoggingAnyExceptionHandler()
                    ))
                    .setResultSerializers(Cf.list(
                            new SimpleResultSerializer(),
                            new SuppressInvocationInfoJsonPojoResultSerializer(mapper),
                            new JsonStringResultSerializer()
                    ))
                    .setDispatcherInterceptors(Cf.list(
                            new ConnectionIdActionDispatcherInterceptor()
                    ))
                    .setInvocationInterceptors(Cf.list(
                            new ThreadLocalCacheInterceptor(),
                            new ResponseInterceptor(),
                            new MasterSlavePolicyInterceptor()
                    ))
                    .setParameterBinders(Cf.list(
                            new BenderParameterBinder(mapper),
                            new BenderJsonListParameterBinder(mapper)
                    ))
                    .setResultType(MediaType.APPLICATION_JSON);

            return ActionConfigurator.configure(actionContainers.plus(cloneableActions), builder);
        }

        @Bean
        public DjfsMigrationActions djfsMigrationActions(BazingaTaskManager bazingaTaskManager,
                DjfsMigratorYtSupplier migratorYtSupplier) {
            return new DjfsMigrationActions(bazingaTaskManager, migratorYtSupplier);
        }
    }

    @Configuration
    public static class Bazinga {
        @Bean
        public BazingaWorkerTaskQueues taskQueues(
                @Value("${djfs-migrator.migrate.tasks.count}") int migrateThreadCount,
                @Value("${djfs-migrator.migrate.tasks.queue}") int migrateQueueCount,
                @Value("${djfs-migrator.cleanup.tasks.count}") int cleanUpThreadCount,
                @Value("${djfs-migrator.cleanup.tasks.queue}") int cleanUpQueueCount,
                @Value("${djfs-migrator.yt-supply.tasks.count}") int ytSupportThreadCount,
                @Value("${djfs-migrator.yt-supply.tasks.queue}") int ytSupportQueueCount
        )
        {
            ListF<TaskQueue> migrateQueue = Cf.range(0, DjfsMigratorTaskQueueName.COPYING_QUEUES_COUNT)
                    .map(DjfsMigratorTaskQueueName::copying)
                    .map(queueName -> new TaskQueue(queueName, migrateThreadCount, migrateQueueCount));
            TaskQueue cleanUpQueue = new TaskQueue(DjfsMigratorTaskQueueName.CLEAN_UP, cleanUpThreadCount, cleanUpQueueCount);
            TaskQueue ytSupport = new TaskQueue(DjfsMigratorTaskQueueName.YT_SUPPORT, ytSupportThreadCount, ytSupportQueueCount);

            return new BazingaWorkerTaskQueues(
                    DjfsMigratorTaskQueueName.CRON, DjfsMigratorTaskQueueName.REGULAR, DjfsMigratorTaskQueueName.CPU_INTENSIVE,
                    migrateQueue.plus(cleanUpQueue, ytSupport));
        }

        @Bean
        public static BeanDefinitionRegistryPostProcessor djfsMigratorCopyTasksRegister() {
            return new BeanDefinitionRegistryPostProcessor() {
                @Override
                public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
                    Cf.range(0, DjfsMigratorTaskQueueName.COPYING_QUEUES_COUNT)
                            .map(DjfsMigratorTaskQueueName::copying)
                            .forEach(queueName -> {
                                GenericBeanDefinition definition = createDefinition(registry, queueName);
                                registry.registerBeanDefinition("DjfsMigratorCopyTask" + queueName, definition);
                            });
                }

                @NotNull
                private GenericBeanDefinition createDefinition(BeanDefinitionRegistry registry, TaskQueueName queueName) {
                    GenericBeanDefinition definition = new GenericBeanDefinition();
                    definition.setBeanClass(DjfsMigratorCopyTask.class);
                    ConstructorArgumentValues constructorArgument = new ConstructorArgumentValues();
                    constructorArgument.addIndexedArgumentValue(0, registry.getBeanDefinition("djfsMigrator"));
                    constructorArgument.addIndexedArgumentValue(1, registry.getBeanDefinition("bazingaTaskManager"));
                    constructorArgument.addIndexedArgumentValue(2, queueName);
                    definition.setConstructorArgumentValues(constructorArgument);
                    return definition;
                }

                @Override
                public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {

                }
            };
        }

        @Bean
        public DjfsMigratorCleanTask djfsMigratorCleanTask(DjfsMigrator djfsMigrator) {
            return new DjfsMigratorCleanTask(djfsMigrator);
        }

        @Bean
        public DjfsMigratorCleanUpLocksTask djfsMigratorCleanUpLocksTask(LockManager lockManager,
                SharpeiClient sharpeiClient) {
            return new DjfsMigratorCleanUpLocksTask(lockManager, sharpeiClient);
        }

        @Bean
        public DjfsMigratorYtSupplyTask djfsMigratorYtSupplyTask(DjfsMigratorYtSupplier djfsMigratorYtSupplier) {
            return new DjfsMigratorYtSupplyTask(djfsMigratorYtSupplier);
        }
    }
}
