package ru.yandex.chemodan.app.djfs.migrator;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.migrator.tasks.DjfsMigratorCopyTask;
import ru.yandex.chemodan.app.djfs.migrator.tasks.DjfsMigratorYtSupplyTask;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class DjfsMigratorYtSupplier {
    private static final Logger logger = LoggerFactory.getLogger(DjfsMigratorYtSupplier.class);
    private static final Duration YT_SUPPLY_INTERVAL = Duration.standardHours(6);
    private static final int THREADS_TO_SUBMIT_TASKS_COUNT = 40;
    private static final int EXPECTED_READY_TASKS_COUNT = 1_000_000;

    private final YtHelper ytHelper;
    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaStorage bazingaStorage;

    public void scheduleSupply(String path) {
        bazingaTaskManager.schedule(new DjfsMigratorYtSupplyTask(path, 0));
    }

    public void addUsersFromYt(String path, long lowerIndex) {
        YPath yPath = YPath.simple(path);
        long tableSize = getRowsCount(yPath);

        int readyTasks = Cf.range(0, DjfsMigratorTaskQueueName.COPYING_QUEUES_COUNT)
                .map(shardId -> DjfsMigratorTaskQueueName.toTaskId(DjfsMigratorTaskQueueName.copying(shardId)))
                .stream()
                .mapToInt(taskId -> bazingaStorage.findOnetimeJobCount(taskId, JobStatus.READY))
                .sum();

        if (readyTasks < EXPECTED_READY_TASKS_COUNT) {
            int rowsToImport = EXPECTED_READY_TASKS_COUNT - readyTasks;
            long upperIndex = lowerIndex + rowsToImport;
            logger.info("Going to import {} users from {} (offset {})", rowsToImport, path, lowerIndex);
            migrate(yPath, lowerIndex, upperIndex);
            if (lowerIndex + rowsToImport < tableSize) {
                logger.info("Scheduling import from {} with offset {})", path, upperIndex);
                scheduleNextImportFromYt(path, upperIndex);
            } else {
                logger.info("All users from {} schduled to migrate)", path);
            }
        } else {
            logger.info("There is enough ready tasks in bazinga {}. Will retry later", readyTasks);
            scheduleNextImportFromYt(path, lowerIndex);
        }
    }

    private void scheduleNextImportFromYt(String path, long offset) {
        Instant nextTaskInstant = Instant.now().plus(YT_SUPPLY_INTERVAL);
        bazingaTaskManager.schedule(new DjfsMigratorYtSupplyTask(path, offset), nextTaskInstant);
    }

    private void migrate(YPath path, long lowerIndex, long upperIndex) {
        YPath pathWithRange = path.withRange(lowerIndex, upperIndex);

        ExecutorService service = Executors.newFixedThreadPool(THREADS_TO_SUBMIT_TASKS_COUNT);

        ytHelper.tables().read(pathWithRange, YTableEntryTypes.bender(MigrationSetup.class), migrationSetup -> {
            logger.info("scheduling task {}", migrationSetup);
            service.submit(() -> scheduleCopyTask(migrationSetup));
        });

        service.shutdown();
    }

    private void scheduleCopyTask(MigrationSetup migrationSetup) {
        bazingaTaskManager.schedule(new DjfsMigratorCopyTask(
                DjfsUid.cons(migrationSetup.uid), migrationSetup.fromShardId, migrationSetup.toShardId, false
        ));
    }

    private long getRowsCount(YPath path) {
        return ytHelper.getRowCount(path);
    }

    @AllArgsConstructor
    @Data
    @BenderBindAllFields
    private static class MigrationSetup {
        private final String uid;
        private final Integer fromShardId;
        private final Integer toShardId;
    }
}
