package ru.yandex.chemodan.app.djfs.migrator.migrations;

import java.time.Duration;
import java.time.Instant;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;

import org.springframework.jdbc.core.ColumnMapRowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgCursorUtils;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.migrator.DjfsCopyConfiguration;
import ru.yandex.chemodan.app.djfs.migrator.PgSchema;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class DjfsAdditionalFileLinksMigration implements DjfsTableMigration {
    private static final Logger logger = LoggerFactory.getLogger(DjfsAdditionalFileLinksMigration.class);

    /**
     * Через триггеры существует constraint, который запрещает существование files.is_live_photo = true
     * без наличия additional_file_links с ссылкой на него (через main_file_fid).
     * <p>
     * Поэтому при миграции files мы переносим без флага, в при миграции additional_file_links
     * восстанавливаем флаг в том виде, в котором он был на изначальном шарде
     * <p>
     * Такая реализация поддерживает извратный кейс с files.is_live_photo = null и наличием additional_file_links
     */
    @Override
    public void runCopying(DjfsCopyConfiguration migrationConf, PgSchema databaseSchema,
            Runnable callback)
    {
        JdbcTemplate3 srcShard = migrationConf.srcShardJdbcTemplate();
        JdbcTemplate3 dstShard = migrationConf.dstShardJdbcTemplate();

        Instant begin = Instant.now();
        AtomicReference<Duration> writeTime = new AtomicReference<>(Duration.ZERO);

        PgCursorUtils.queryWithCursorAsBatches(
                srcShard.getDataSource(), migrationConf.getBaseBatchSize(),
                "SELECT * FROM disk.additional_file_links WHERE uid = ?",
                new ColumnMapRowMapper(),
                migrationConf.getUid().asLong()
        ).forEach(fetchedRows -> {
            ListF<UUID> withIsLivePhotoOnSrc = srcShard.query(
                    "SELECT fid, is_live_photo from disk.files WHERE uid = :uid AND fid IN (:fids)",
                    (rs, i) -> Tuple2.tuple(
                            ResultSetUtils.getUuid(rs, "fid"),
                            ResultSetUtils.getBooleanO(rs, "is_live_photo")
                    ),
                    Cf.map(
                            "uid", migrationConf.getUid(),
                            "fids", fetchedRows.map(row -> row.get("main_file_fid"))
                    )
            )
                    .filter(t -> t.get2().getOrElse(false))
                    .map(Tuple2::get1);

            Instant beginWrite = Instant.now();
            DjfsMigrationUtil.withDisabledMigrationLockCheck(dstShard, () -> {
                if (withIsLivePhotoOnSrc.isNotEmpty()) {
                    dstShard.update(
                            "UPDATE disk.files SET is_live_photo = true WHERE uid = :uid and fid IN (:fids)",
                            Cf.map(
                                    "uid", migrationConf.getUid(),
                                    "fids", withIsLivePhotoOnSrc
                            )
                    );
                }
                DjfsMigrationUtil.copyRows(dstShard, databaseSchema, "additional_file_links", fetchedRows);
            });
            writeTime.accumulateAndGet(Duration.between(beginWrite, Instant.now()), Duration::plus);
            callback.run();
        });
        Duration overallTime = Duration.between(begin, Instant.now());
        logger.info("overallTime {}, read time {}, write time {}",
                overallTime, overallTime.minus(writeTime.get()), writeTime.get()
        );
    }

    @Override
    public void checkAllCopied(DjfsCopyConfiguration migrationConf, PgSchema pgSchema) {
        JdbcTemplate3 srcShard = migrationConf.srcShardJdbcTemplate();
        JdbcTemplate3 dstShard = migrationConf.dstShardJdbcTemplate();

        DjfsUid uid = migrationConf.getUid();

        DjfsMigrationUtil.checkSameDataInTableForUid(srcShard, dstShard, "additional_file_links", uid, pgSchema);
    }

    @Override
    public void cleanData(JdbcTemplate3 shard, DjfsUid uid, int batchSize) {
        DjfsMigrationUtil.withDisabledMigrationLockCheck(shard, () ->
                shard.update(""
                                + "WITH to_delete(uid, main_file_fid) AS ("
                                + "     SELECT uid, main_file_fid FROM disk.additional_file_links WHERE uid = ?"
                                + "),"
                                + "update_files AS ("
                                + "     UPDATE disk.files SET is_live_photo = NULL "
                                + "         WHERE (uid, fid) IN (SELECT uid, main_file_fid from to_delete)"
                                + ")"
                                + "DELETE FROM disk.additional_file_links "
                                + "     WHERE (uid, main_file_fid) IN (SELECT uid, main_file_fid from to_delete)",
                        uid.asLong()
                )
        );
    }

    @Override
    public ListF<String> tables() {
        return Cf.list("additional_file_links");
    }
}
