package ru.yandex.chemodan.app.djfs.migrator.migrations;

import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;

@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
@Getter
public class DjfsMigrationPlan {
    private final ListF<DjfsTableMigration> migrations;
    private final ListF<String> tablesOrder;
    private final ListF<String> selfReference;
    private final ListF<String> ignore;

    public static DjfsMigrationPlan allTablesMigrations = builder()
            .migration(new DjfsAlbumsMigration())

            .simple("changelog")

            .simple("deletion_log")

            .withSelfReference("disk_info", "id", "parent")


            .migration(new DjfsFoldersMigration())
            .simple("version_links")
            .migration(new DjfsFilesDataMigration())
            .migration(new DjfsAdditionalFileLinksMigration())

            .simple("filesystem_locks")

            .withSelfReference("link_data", "id", "parent")

            .withSelfReference("misc_data", "id", "parent")

            .simple("operations")

            .simple("source_ids")

            .simple("user_index")
            .simple("user_activity_info")

            .simple("album_deltas")
            .simple("albums_info")
            .simple("album_face_clusters")

            .ignore("task_queue", "very new")

            .ignore("group_invites", "not used")
            .ignore("group_links", "not used")
            .ignore("groups", "not used")

            .cleanButNotCopy("async_tasks_data")
            .cleanButNotCopy("last_files_cache")
            .ignore("resources_index", "self organized by triggers")
            .ignore("migration_lock", "special logic")
            .build();

    public static DjfsMigrationPlan.Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private final ListF<DjfsTableMigration> migrations = Cf.arrayList();
        private final ListF<String> tablesOrder = Cf.arrayList();
        private final ListF<String> selfReference = Cf.arrayList();
        private final ListF<String> ignore = Cf.arrayList();

        Builder simple(String tableName) {
            return migration(new DjfsSimpleTableMigration(tableName));
        }

        Builder cleanButNotCopy(String tableName) {
            return migration(new DjfsCleanButNotCopyTableMigration(tableName));
        }

        public Builder migration(DjfsTableMigration migration) {
            if (migration instanceof DjfsTableWithSelfReferenceMigration) {
                selfReference.addAll(migration.tables());
            }
            migrations.add(migration);
            tablesOrder.addAll(migration.tables());
            return this;
        }

        Builder withSelfReference(String tableName, String referenceFrom, String referenceTo) {
            return migration(new DjfsTableWithSelfReferenceMigration(tableName, referenceFrom, referenceTo));
        }

        Builder ignore(String tableName, /*just for documentation*/ String reason) {
            ignore.add(tableName);
            return this;
        }

        public DjfsMigrationPlan build() {
            return new DjfsMigrationPlan(
                    migrations.makeReadOnly(),
                    tablesOrder.makeReadOnly(),
                    selfReference.makeReadOnly(),
                    ignore.makeReadOnly()
            );
        }
    }
}
