package ru.yandex.chemodan.app.djfs.migrator.migrations;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.atomic.AtomicReference;

import org.springframework.jdbc.core.ColumnMapRowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgCursorUtils;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.migrator.DjfsCopyConfiguration;
import ru.yandex.chemodan.app.djfs.migrator.PgSchema;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.test.Assert;

/**
 * @author yappo
 */
public class DjfsSimpleTableMigration implements DjfsTableMigration {
    private static final Logger logger = LoggerFactory.getLogger(DjfsSimpleTableMigration.class);

    private final String tableName;

    public DjfsSimpleTableMigration(String tableName) {
        Assert.notEmpty(tableName, "table name is empty");
        this.tableName = tableName;
    }

    @Override
    public void runCopying(DjfsCopyConfiguration migrationConf, PgSchema databaseSchema,
            Runnable callback)
    {
        JdbcTemplate3 srcShard = migrationConf.srcShardJdbcTemplate();
        JdbcTemplate3 dstShard = migrationConf.dstShardJdbcTemplate();

        Instant begin = Instant.now();
        AtomicReference<Duration> writeTime = new AtomicReference<>(Duration.ZERO);

        PgCursorUtils.queryWithCursorAsBatches(
                srcShard.getDataSource(), migrationConf.getBaseBatchSize(),
                "SELECT * FROM disk." + tableName + " WHERE uid = ?",
                new ColumnMapRowMapper(),
                migrationConf.getUid().asLong()
        ).forEach(fetchedRows -> {
            Instant beginWrite = Instant.now();
            DjfsMigrationUtil.copyRows(dstShard, databaseSchema, tableName, fetchedRows);
            writeTime.accumulateAndGet(Duration.between(beginWrite, Instant.now()), Duration::plus);
            callback.run();
        });
        Duration overallTime = Duration.between(begin, Instant.now());
        logger.info("overallTime {}, read time {}, write time {}",
                overallTime, overallTime.minus(writeTime.get()), writeTime.get()
        );
    }

    @Override
    public void checkAllCopied(DjfsCopyConfiguration migrationConf,
            PgSchema sourceSchema)
    {
        JdbcTemplate3 srcShard = migrationConf.srcShardJdbcTemplate();
        JdbcTemplate3 dstShard = migrationConf.dstShardJdbcTemplate();

        DjfsMigrationUtil.checkSameDataInTableForUid(srcShard, dstShard, tableName, migrationConf.getUid(),
                sourceSchema);
    }

    @Override
    public void cleanData(JdbcTemplate3 shard, DjfsUid uid, int batchSize) {
        DjfsMigrationUtil.withDisabledMigrationLockCheck(shard, () ->
                shard.update("DELETE FROM disk." + tableName + " WHERE uid = ?", uid.asLong())
        );
    }

    @Override
    public ListF<String> tables() {
        return Cf.list(tableName);
    }

}
