package ru.yandex.chemodan.app.djfs.migrator.tasks;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigrationState;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigrator;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigratorEvents;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigratorTaskQueueName;
import ru.yandex.chemodan.app.djfs.migrator.migrations.DjfsMigrationPlan;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class DjfsMigratorCleanTask extends OnetimeTaskSupport<DjfsMigratorCleanTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(DjfsMigratorCleanTask.class);

    private final DjfsMigrator djfsMigrator;

    public DjfsMigratorCleanTask(DjfsUid uid, int fromShardId, int toShardId) {
        super(new Parameters(uid.asString(), fromShardId, toShardId));
        djfsMigrator = null;
    }

    public DjfsMigratorCleanTask(DjfsMigrator djfsMigrator)
    {
        super(DjfsMigratorCleanTask.Parameters.class);
        this.djfsMigrator = djfsMigrator;
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) {
        DjfsMigratorFancy.checkPermission();
        try {
            Instant begin = Instant.now();
            DjfsMigratorEvents.logEvent(
                    parameters.getRowUid(), parameters.getFromShardId(), parameters.getToShardId(),
                    DjfsMigrationState.START_CLEANING, "user cleaning is started"
            );
            djfsMigrator.cleanData(
                    DjfsUid.cons(parameters.getRowUid()), parameters.getFromShardId(),
                    Cf.list(parameters.getFromShardId(), parameters.getToShardId()),
                    DjfsMigrationPlan.allTablesMigrations
            );
            DjfsMigratorEvents.logEvent(
                    parameters.getRowUid(), parameters.getFromShardId(), parameters.getToShardId(),
                    DjfsMigrationState.SUCCESS_CLEAN, "user clean is success. Spend time " + new Duration(begin, Instant.now())
            );
        } catch (Throwable e) {
            logger.error("error on clean task {}", parameters, e);
            DjfsMigratorEvents.logEvent(
                    parameters.getRowUid(), parameters.getFromShardId(), parameters.getToShardId(),
                    DjfsMigrationState.EXCEPTION_ON_CLEANING, e.getMessage()
            );
            throw e;
        }
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsMigratorTaskQueueName.CLEAN_UP;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(1);
    }

    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final String rowUid;
        private final int fromShardId;
        private final int toShardId;
    }
}
