package ru.yandex.chemodan.app.djfs.migrator.tasks;

import java.util.Optional;

import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigrator;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigratorTaskQueueName;
import ru.yandex.chemodan.util.sharpei.SharpeiClient;
import ru.yandex.chemodan.util.sharpei.SharpeiShardInfo;
import ru.yandex.chemodan.util.sharpei.SharpeiUserInfo;
import ru.yandex.chemodan.util.sharpei.SimpleUserId;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

@RequiredArgsConstructor
public class DjfsMigratorCleanUpLocksTask extends CronTask {
    private static final Logger logger = LoggerFactory.getLogger(DjfsMigratorCleanUpLocksTask.class);

    private final LockManager lockManager;
    private final SharpeiClient sharpeiClient;
    @Override
    public Schedule cronExpression() {
        return ScheduleCron.parse("0 3 * * *", TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
    }

    @Override
    public void execute(ExecutionContext executionContext) throws Exception {
        Optional<Exception> exception = Optional.empty();
        Option<Integer> commonShardId = sharpeiClient.findUser(new SimpleUserId(DjfsUid.COMMON_UID.asString()))
                .map(SharpeiUserInfo::getShard)
                .map(SharpeiShardInfo::getId);
        for (SharpeiShardInfo shard : sharpeiClient.getShards().filterNot((s) -> commonShardId.isSome(s.getId()))) {
            try {
                lockManager.cleanUpMigrationLocks(
                        new DjfsShardInfo.Pg(shard.getId()),
                        Instant.now().minus(DjfsMigrator.MigrationLock.LOCK_SOURCE_SHARD_AFTER_MIGRATION_FOR)
                );
            } catch (Exception e) {
                exception = Optional.of(e);
                logger.error("exception on clean up locks on {}", shard, e);
            }
        }
        if (exception.isPresent()) {
            throw exception.get();
        }
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsMigratorTaskQueueName.CRON;
    }
}
