package ru.yandex.chemodan.app.djfs.migrator.tasks;

import java.util.concurrent.ThreadLocalRandom;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigrationState;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigrator;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigratorEvents;
import ru.yandex.chemodan.app.djfs.migrator.DjfsMigratorTaskQueueName;
import ru.yandex.chemodan.app.djfs.migrator.migrations.DjfsMigrationPlan;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class DjfsMigratorCopyTask extends OnetimeTaskSupport<DjfsMigratorCopyTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(DjfsMigratorCopyTask.class);

    private final DjfsMigrator djfsMigrator;
    private final BazingaTaskManager bazingaTaskManager;
    private final TaskQueueName queueName;

    public DjfsMigratorCopyTask(DjfsUid uid, int fromShardId, int toShardId, boolean forceActiveUser) {
        super(Parameters.builder()
                .rowUid(uid.asString())
                .fromShardId(fromShardId)
                .toShardId(toShardId)
                .forceActiveUser(forceActiveUser)
                .build());
        djfsMigrator = null;
        bazingaTaskManager = null;
        queueName = DjfsMigratorTaskQueueName.copying(toShardId);
    }

    private DjfsMigratorCopyTask(Parameters parameters) {
        super(parameters);
        djfsMigrator = null;
        bazingaTaskManager = null;
        queueName = DjfsMigratorTaskQueueName.copying(parameters.toShardId);
    }

    //used by ru.yandex.chemodan.app.djfs.migrator.DjfsMigratorApplicationContext.Bazinga.djfsMigratorCopyTasksRegister
    @SuppressWarnings("unused")
    public DjfsMigratorCopyTask(DjfsMigrator djfsMigrator, BazingaTaskManager bazingaTaskManager, TaskQueueName queueName)
    {
        super(DjfsMigratorCopyTask.Parameters.class);
        this.djfsMigrator = djfsMigrator;
        this.bazingaTaskManager = bazingaTaskManager;
        this.queueName = queueName;
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) {
        try {
            DjfsMigratorFancy.checkPermission();

            DjfsUid uid = DjfsUid.cons(parameters.rowUid);
            int destinationShardId = parameters.toShardId;

            Instant begin = Instant.now();
            DjfsMigratorEvents.logEvent(
                    parameters.getRowUid(), parameters.getFromShardId(), parameters.getToShardId(),
                    DjfsMigrationState.START_COPYING, "user migration is started"
            );
            DjfsMigrator.CopyResult result = djfsMigrator.copyDataAndSwitchShard(
                    uid, parameters.fromShardId, destinationShardId,
                    parameters.forceActiveUser, DjfsMigrationPlan.allTablesMigrations
            );
            DjfsMigratorEvents.logEvent(
                    parameters.getRowUid(), parameters.getFromShardId(), parameters.getToShardId(),
                    result.getState(), result.getMessage() + " Spend time " + new Duration(begin, Instant.now())
            );
            switch (result.getState()) {
                case COPY_SUCCESS:
                    scheduleCleaning(uid, parameters.getFromShardId(), parameters.getToShardId());
                    break;
                case USER_IS_ACTIVE:
                    retryUserIsActive(parameters);
                    break;
                case ALREADY_MIGRATED:
                    //nothing to do
                    break;
                case EXCEPTION_ON_COPYING:
                case USER_ALREADY_LOCKED_FOR_MIGRATION:
                case MIGRATION_CONFIG_ERROR:
                    retryError(parameters);
                    break;
                default:
                    throw new UnsupportedOperationException(
                            result.getState().isCopyResult() ? "must be handled" : "must not returned by djfsMigrator.copyData"
                    );
            }
        } catch (Throwable e) {
            logger.error("error on copy task {}", parameters, e);
            DjfsMigratorEvents.logEvent(
                    parameters.getRowUid(), parameters.getFromShardId(), parameters.getToShardId(),
                    DjfsMigrationState.EXCEPTION_ON_COPYING, e.getMessage()
            );
            throw e;
        }
    }

    private void scheduleCleaning(DjfsUid uid, int fromShardId, int toShardId) {
        bazingaTaskManager.schedule(
                new DjfsMigratorCleanTask(uid, fromShardId, toShardId),
                Instant.now().plus(DjfsMigrator.MigrationLock.LOCK_SOURCE_SHARD_AFTER_MIGRATION_FOR).plus(Duration.standardHours(1))
        );
    }

    private void retryError(Parameters parameters) {
        Parameters newParameters = parameters.copy().errorRetryCount(parameters.errorRetryCount + 1).build();
        if (newParameters.errorRetryCount > 10) {
            logger.error("errorRetryCount exceeded {}", parameters);
            return;
        }
        Duration after = Duration.standardMinutes(
                ThreadLocalRandom.current().nextInt(newParameters.errorRetryCount, 5 * newParameters.errorRetryCount)
        );

        bazingaTaskManager.schedule(new DjfsMigratorCopyTask(newParameters), Instant.now().plus(after));
    }

    private void retryUserIsActive(Parameters parameters) {
        Parameters newParameters = parameters.copy().userActiveCount(parameters.userActiveCount + 1).build();
        if (newParameters.userActiveCount > 10) {
            logger.error("userActiveCount exceeded {}", parameters);
            // alternative solution is:
            // newParameters = newParameters.copy().forceActiveUser(true).build();
            return;
        }
        Duration after = Duration.standardMinutes(
                60 * ThreadLocalRandom.current().nextInt(newParameters.userActiveCount, 2 * newParameters.userActiveCount)
        );

        bazingaTaskManager.schedule(new DjfsMigratorCopyTask(newParameters), Instant.now().plus(after));
    }

    @Override
    public TaskQueueName queueName() {
        return queueName;
    }

    @Override
    public TaskId id() {
        return DjfsMigratorTaskQueueName.toTaskId(queueName);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(2);
    }

    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    @Builder(builderClassName = "Builder")
    public static class Parameters {
        private final String rowUid;
        private final int fromShardId;
        private final int toShardId;
        //TODO for future use
        private final boolean forceActiveUser;
        private final int errorRetryCount;
        private final int userActiveCount;

        public Builder copy() {
            return builder()
                    .rowUid(rowUid)
                    .fromShardId(fromShardId)
                    .toShardId(toShardId)
                    .forceActiveUser(forceActiveUser)
                    .errorRetryCount(errorRetryCount)
                    .userActiveCount(userActiveCount);
        }
    }
}
