package ru.yandex.chemodan.app.djfs.albums;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.albums.indexer.YtFetchCoordinatesSubmitTask;
import ru.yandex.chemodan.app.djfs.core.album.AlbumItemFieldValue;
import ru.yandex.chemodan.app.djfs.core.album.AlbumItemFieldValueJsonSerializers;
import ru.yandex.chemodan.app.djfs.core.album.AlbumMergeManager;
import ru.yandex.chemodan.app.djfs.core.album.CatchupAlbumCreationHandler;
import ru.yandex.chemodan.app.djfs.core.album.DeleteFacesTask;
import ru.yandex.chemodan.app.djfs.core.album.FacesAlbumManager;
import ru.yandex.chemodan.app.djfs.core.album.FetchAestheticsForMauFromIndexerTask;
import ru.yandex.chemodan.app.djfs.core.album.FetchCoordinatesFromIndexerTask;
import ru.yandex.chemodan.app.djfs.core.album.FileOperationPostProcessHandler;
import ru.yandex.chemodan.app.djfs.core.album.FileOperationPostProcessingTask;
import ru.yandex.chemodan.app.djfs.core.album.GenerateGeoAlbumsTask;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumCoverUpdateTask;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumManager;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumResourceProcessingTask;
import ru.yandex.chemodan.app.djfs.core.album.PersonalAlbumManager;
import ru.yandex.chemodan.app.djfs.core.album.RawJsonValue;
import ru.yandex.chemodan.app.djfs.core.album.RawJsonValueSerializer;
import ru.yandex.chemodan.app.djfs.core.album.ReinitializeUserFacesTask;
import ru.yandex.chemodan.app.djfs.core.album.SynchronizedToAlbumLockedInterceptor;
import ru.yandex.chemodan.app.djfs.core.album.UpdateUserFacesTask;
import ru.yandex.chemodan.app.djfs.core.album.operation.albummerge.AlbumMergeTask;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsQueueTask;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsQueueWatchdogTask;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsTaskDao;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsTaskManager;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchHttpClient;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfoManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.index.BazingaManagerProperties;
import ru.yandex.chemodan.app.djfs.core.legacy.web.LegacyMpfsExceptionHandler;
import ru.yandex.chemodan.app.djfs.core.operations.OperationDao;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsAlbumsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.CheckBlockedInterceptor;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.web.ConnectionIdActionDispatcherInterceptor;
import ru.yandex.chemodan.app.djfs.core.web.DjfsTskvLog4jRequestLog;
import ru.yandex.chemodan.app.djfs.core.web.JsonStringResultSerializer;
import ru.yandex.chemodan.app.djfs.core.web.LoggingAnyExceptionHandler;
import ru.yandex.chemodan.app.djfs.core.web.UidActionDispatcherInterceptor;
import ru.yandex.chemodan.bazinga.BazingaWorkerTaskQueues;
import ru.yandex.chemodan.boot.ChemodanCommonContextConfiguration;
import ru.yandex.chemodan.boot.admin.ChemodanAdminDaemonContextConfiguration;
import ru.yandex.chemodan.queller.worker.BazingaBeansContextConfiguration;
import ru.yandex.chemodan.queller.worker.CeleryOnetimeTask;
import ru.yandex.chemodan.util.web.A3JettyConfiguration;
import ru.yandex.chemodan.util.web.A3JettyContextConfiguration;
import ru.yandex.chemodan.util.web.MasterSlaveUnavailableExceptionHandler;
import ru.yandex.chemodan.util.web.SuppressInvocationInfoJsonPojoResultSerializer;
import ru.yandex.chemodan.util.web.interceptors.ThreadLocalCacheInterceptor;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.a3.ActionApp;
import ru.yandex.commune.a3.ActionConfigurationBuilder;
import ru.yandex.commune.a3.ActionConfigurator;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.parameter.bind.BenderJsonListParameterBinder;
import ru.yandex.commune.a3.action.parameter.bind.BenderParameterBinder;
import ru.yandex.commune.a3.action.parameter.convert.DefaultConverters;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.commune.a3.action.result.ResponseInterceptor;
import ru.yandex.commune.a3.action.result.SimpleResultSerializer;
import ru.yandex.commune.a3.action.result.error.A3ExceptionHandler;
import ru.yandex.commune.a3.action.result.error.HttpStatusCodeSourceExceptionHandler;
import ru.yandex.commune.a3.action.result.error.nested.ExceptionWithRootElementHandler;
import ru.yandex.commune.a3.action.result.type.MediaType;
import ru.yandex.commune.a3.db.MasterSlavePolicyInterceptor;
import ru.yandex.commune.a3.security.SecurityExceptionHandler;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.TaskOverridesManager;
import ru.yandex.commune.bazinga.impl.worker.WorkerTaskRegistry;
import ru.yandex.commune.bazinga.pg.worker.PgBazingaWorkerConfiguration;
import ru.yandex.commune.bazinga.scheduler.TaskQueue;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.commune.json.bender.JsonValueMarshaller;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.ReadableInstantConfigurableMarshaller;
import ru.yandex.misc.spring.ApplicationContextUtils;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;

@Configuration
@Import({
        A3JettyContextConfiguration.class,
        ChemodanCommonContextConfiguration.class,
        ChemodanAdminDaemonContextConfiguration.class,
        BazingaBeansContextConfiguration.class,
})
public abstract class DjfsAlbumsBaseContextConfiguration {
    @Bean
    public A3JettyConfiguration a3JettyServletsConfiguration() {
        return new A3JettyConfiguration(Tuple2List.fromPairs("/api/*", "")) {
            @Override
            public void postConstruct(SingleWarJetty jetty, ActionApp actionApp) {
                jetty.setRequestLogFactory(DjfsTskvLog4jRequestLog::new);
            }
        };
    }

    @Bean
    public ActionApp actionApp(ApplicationContext context, ApplicationInfo applicationInfo, UserDao userDao) {
        BenderMapper mapper = new BenderMapper(BenderConfiguration.cons(
                MembersToBind.WITH_ANNOTATIONS, false,
                CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(Instant.class, new ReadableInstantConfigurableMarshaller(ISODateTimeFormat.dateTime()))
                        .add(AlbumItemFieldValue.class, AlbumItemFieldValueJsonSerializers.consMarshaller())
                        .add(JsonValue.class, new JsonValueMarshaller())
                        .add(RawJsonValue.class, RawJsonValueSerializer.consMarshaller())
                        .build()
        ));

        ListF<CloneableAction> cloneableActions = ApplicationContextUtils.beansOfType(context, CloneableAction.class);

        ListF<Object> actionContainers =
                ApplicationContextUtils.beansWithAnnotationList(context, ActionContainer.class).get2();

        ActionConfigurationBuilder builder = ActionConfigurationBuilder
                .cons(applicationInfo)
                .setConverters(DefaultConverters.all().getConverters())
                .setExceptionHandlers(Cf.list(
                        context.getBean(LegacyMpfsExceptionHandler.class),
                        new MasterSlaveUnavailableExceptionHandler(),
                        new HttpStatusCodeSourceExceptionHandler(),
                        new ExceptionWithRootElementHandler(),
                        new SecurityExceptionHandler(),
                        new A3ExceptionHandler(),
                        new LoggingAnyExceptionHandler()
                ))
                .setResultSerializers(Cf.list(
//                        new JsonPojoResultSerializer(mapper2),
                        new SimpleResultSerializer(),
                        new SuppressInvocationInfoJsonPojoResultSerializer(mapper),
                        new JsonStringResultSerializer()
                ))
                .setDispatcherInterceptors(Cf.list(
                        new UidActionDispatcherInterceptor(),
                        new ConnectionIdActionDispatcherInterceptor()
                ))
                .setInvocationInterceptors(Cf.list(
                        new ThreadLocalCacheInterceptor(),
                        new ResponseInterceptor(),
                        new CheckBlockedInterceptor(userDao),
                        new SynchronizedToAlbumLockedInterceptor(),
                        new MasterSlavePolicyInterceptor()
                ))
                .setParameterBinders(Cf.list(
                        new BenderParameterBinder(mapper),
                        new BenderJsonListParameterBinder(mapper)
                ))
                .setResultType(MediaType.APPLICATION_JSON);

        return ActionConfigurator.configure(actionContainers.plus(cloneableActions), builder);
    }

    @Bean
    public PgBazingaWorkerConfiguration pgBazingaWorkerConfiguration() {
        return new PgBazingaWorkerConfiguration(Cf.list(CeleryOnetimeTask.BAZINGA_QUEUE_NAME));
    }

    @Bean
    public BazingaWorkerTaskQueues bazingaWorkerTaskQueues(
            @Value("${djfs-albums.geo-tasks-queue.count}") int geoTasksQueueThreadCount,
            @Value("${djfs-albums.geo-tasks-queue.queue}") int geoTasksQueueCount,
            @Value("${djfs-albums.indexer-tasks-queue.count}") int indexerTasksQueueThreadCount,
            @Value("${djfs-albums.indexer-tasks-queue.queue}") int indexerTasksQueueCount,
            @Value("${djfs-albums.geo-generation-tasks-queue.count}") int geoGenerationTasksQueueThreadCount,
            @Value("${djfs-albums.geo-generation-tasks-queue.queue}") int geoGenerationTasksQueueCount,
            @Value("${djfs-albums.merge-tasks-queue.count}") int mergeTasksQueueThreadCount,
            @Value("${djfs-albums.merge-tasks-queue.queue}") int mergeTasksQueueCount,
            @Value("${djfs-albums.reinitialize-user-faces-tasks-queue.count}") int reinitializeUserFacesTasksQueueThreadCount,
            @Value("${djfs-albums.reinitialize-user-faces-tasks-queue.queue}") int reinitializeUserFacesTasksQueueCount,
            @Value("${djfs-albums.update-user-faces-tasks-queue.count}") int updateUserFacesTasksQueueThreadCount,
            @Value("${djfs-albums.update-user-faces-tasks-queue.queue}") int updateUserFacesTasksQueueCount,
            @Value("${djfs-albums.queue-tasks-queue.count}") int queueTasksQueueThreadCount,
            @Value("${djfs-albums.queue-tasks-queue.queue}") int queueTasksQueueCount,
            @Value("${djfs-albums.delete-user-faces-tasks-queue.count}") int deleteUserFacesTasksQueueThreadCount,
            @Value("${djfs-albums.delete-user-faces-tasks-queue.queue}") int deleteUserFacesTasksQueueCount)
    {
        TaskQueue geoTasksQueue = new TaskQueue(DjfsAlbumsTaskQueueName.ALBUMS_GEO_TASKS,
                geoTasksQueueThreadCount, geoTasksQueueCount);
        TaskQueue indexerTasksQueue = new TaskQueue(DjfsAlbumsTaskQueueName.ALBUMS_INDEXER_TASKS,
                indexerTasksQueueThreadCount, indexerTasksQueueCount);
        TaskQueue geoGenerationTasksQueue = new TaskQueue(DjfsAlbumsTaskQueueName.ALBUMS_GEO_GENERATION_TASKS,
                geoGenerationTasksQueueThreadCount, geoGenerationTasksQueueCount);
        TaskQueue mergeTasksQueue = new TaskQueue(DjfsAlbumsTaskQueueName.ALBUMS_MERGE_TASKS,
                mergeTasksQueueThreadCount, mergeTasksQueueCount);
        TaskQueue reinitializeUserFacesTasksQueue =
                new TaskQueue(
                        DjfsAlbumsTaskQueueName.ALBUMS_REINITIALIZE_USER_FACES_TASKS,
                        reinitializeUserFacesTasksQueueThreadCount,
                        reinitializeUserFacesTasksQueueCount
                );
        TaskQueue updateUserFacesTasksQueue =
                new TaskQueue(
                        DjfsAlbumsTaskQueueName.ALBUMS_UPDATE_USER_FACES_TASKS,
                        updateUserFacesTasksQueueThreadCount,
                        updateUserFacesTasksQueueCount
                );
        TaskQueue deleteUserFacesTasksQueue =
                new TaskQueue(
                        DjfsAlbumsTaskQueueName.ALBUMS_DELETE_USER_FACES_TASKS,
                        deleteUserFacesTasksQueueThreadCount,
                        deleteUserFacesTasksQueueCount
                );
        TaskQueue queueTasksQueue = new TaskQueue(DjfsAlbumsTaskQueueName.ALBUMS_QUEUE_TASKS,
                queueTasksQueueThreadCount, queueTasksQueueCount);
        return new BazingaWorkerTaskQueues(TaskQueueName.CRON, TaskQueueName.REGULAR, TaskQueueName.CPU_INTENSIVE,
                Cf.list(
                        new TaskQueue(CeleryOnetimeTask.BAZINGA_QUEUE_NAME, 1, 0),
                        geoTasksQueue,
                        indexerTasksQueue,
                        geoGenerationTasksQueue,
                        mergeTasksQueue,
                        reinitializeUserFacesTasksQueue,
                        updateUserFacesTasksQueue,
                        queueTasksQueue,
                        deleteUserFacesTasksQueue
                )
        );
    }

    @Bean
    public GeoAlbumResourceProcessingTask geoAlbumResourceProcessingTask(Filesystem filesystem,
                                                                         GeoAlbumManager geoAlbumManager, TransactionUtils transactionUtils)
    {
        return new GeoAlbumResourceProcessingTask(filesystem, geoAlbumManager, transactionUtils);
    }

    @Bean
    public GeoAlbumCoverUpdateTask geoAlbumCoverUpdateTask(Filesystem filesystem,
                                                           GeoAlbumManager geoAlbumManager, TransactionUtils transactionUtils)
    {
        return new GeoAlbumCoverUpdateTask(filesystem, geoAlbumManager, transactionUtils);
    }

    @Bean
    public GenerateGeoAlbumsTask generateGeoAlbumsTask(GeoAlbumManager geoAlbumManager) {
        return new GenerateGeoAlbumsTask(geoAlbumManager);
    }

    @Bean
    public FetchCoordinatesFromIndexerTask fetchCoordinatesFromIndexerTask(DjfsResourceDao djfsResourceDao,
                                                                           DiskSearchHttpClient diskSearchHttpClient, UserDao userDao, DiskInfoManager diskInfoManager,
                                                                           BazingaTaskManager bazingaTaskManager)
    {
        return new FetchCoordinatesFromIndexerTask(djfsResourceDao, diskSearchHttpClient, userDao, diskInfoManager,
                bazingaTaskManager);
    }

    @Bean
    public FetchAestheticsForMauFromIndexerTask fetchAestheticsForMauFromIndexerTask(DjfsResourceDao djfsResourceDao,
                                                                                     DiskSearchHttpClient diskSearchHttpClient, UserDao userDao)
    {
        return new FetchAestheticsForMauFromIndexerTask(djfsResourceDao, diskSearchHttpClient, userDao);
    }

    @Bean
    public DjfsAlbumsTaskManager djfsAlbumsTaskManager(BazingaTaskManager bazingaTaskManager, DjfsAlbumsTaskDao djfsAlbumsTaskDao) {
        return new DjfsAlbumsTaskManager(bazingaTaskManager, djfsAlbumsTaskDao);
    }

    @Bean
    public DjfsAlbumsQueueTask djfsAlbumsQueueTask(DjfsAlbumsTaskDao djfsAlbumsTaskDao,
            GeoAlbumCoverUpdateTask geoAlbumCoverUpdateTask, GeoAlbumResourceProcessingTask geoAlbumResourceProcessingTask, FileOperationPostProcessingTask fileOperationPostProcessingTask,
            BazingaTaskManager bazingaTaskManager, BazingaManagerProperties bazingaTaskManagerProperties,
            TaskOverridesManager taskOverridesManager, WorkerTaskRegistry workerTaskRegistry) {
        return new DjfsAlbumsQueueTask(djfsAlbumsTaskDao,
                geoAlbumCoverUpdateTask, geoAlbumResourceProcessingTask, fileOperationPostProcessingTask,
                bazingaTaskManager, bazingaTaskManagerProperties,
                taskOverridesManager, workerTaskRegistry);
    }

    @Bean
    public DjfsAlbumsQueueWatchdogTask djfsAlbumsQueueWatchdogTask(DjfsAlbumsTaskDao djfsAlbumsTaskDao,
                                                                   BazingaTaskManager bazingaTaskManager, BazingaManagerProperties bazingaTaskManagerProperties) {
        return new DjfsAlbumsQueueWatchdogTask(djfsAlbumsTaskDao,
                bazingaTaskManager, bazingaTaskManagerProperties);
    }

    @Bean
    public YtHelper ytHelper(@Value("${yt.url}") String ytUrl, @Value("${djfs.yql.token}") String ytToken) {
        Yt yt = YtUtils.http(ytUrl, ytToken);
        RetryPolicy retryPolicy = new RetryPolicy().withMaxRetries(3).withDelay(10, TimeUnit.SECONDS);
        return new YtHelper(yt, retryPolicy);
    }

    @Bean
    public YtFetchCoordinatesSubmitTask ytFetchCoordinatesSubmitTask(YtHelper ytHelper,
                                                                     BazingaTaskManager bazingaTaskManager)
    {
        return new YtFetchCoordinatesSubmitTask(ytHelper, bazingaTaskManager);
    }

    @Bean
    public FileOperationPostProcessHandler fileOperationPostProcessHandler(DjfsAlbumsTaskManager djfsAlbumsTaskManager,
            GeoAlbumManager geoAlbumManager, PersonalAlbumManager personalAlbumManager)
    {
        return new FileOperationPostProcessHandler(djfsAlbumsTaskManager, geoAlbumManager, personalAlbumManager);
    }

    @Bean
    public FileOperationPostProcessingTask fileOperationPostProcessingTask(GeoAlbumManager geoAlbumManager,
                                                                           PersonalAlbumManager personalAlbumManager)
    {
        return new FileOperationPostProcessingTask(geoAlbumManager, personalAlbumManager);
    }

    @Bean
    public CatchupAlbumCreationHandler catchupAlbumCreationHandler(GeoAlbumManager geoAlbumManager) {
        return new CatchupAlbumCreationHandler(geoAlbumManager);
    }

    @Bean
    public AlbumMergeTask albumMergeTask(AlbumMergeManager albumMergeManager, OperationDao operationDao) {
        return new AlbumMergeTask(albumMergeManager, operationDao);
    }

    @Bean
    public DjfsAlbumsTaskDao djfsAlbumsTaskDao(PgShardedDaoContext context) {
        return new DjfsAlbumsTaskDao(context);
    }

    @Bean
    public ReinitializeUserFacesTask reinitializeUserFacesTask(FacesAlbumManager facesAlbumManager) {
        return new ReinitializeUserFacesTask(facesAlbumManager);
    }

    @Bean
    public UpdateUserFacesTask updateUserFacesTask(FacesAlbumManager facesAlbumManager) {
        return new UpdateUserFacesTask(facesAlbumManager);
    }

    @Bean
    public DeleteFacesTask deleteFacesTask(FacesAlbumManager facesAlbumManager) {
        return new DeleteFacesTask(facesAlbumManager);
    }
}
