package ru.yandex.chemodan.app.djfs.albums.indexer;

import com.fasterxml.jackson.databind.JsonNode;
import org.joda.time.Duration;

import ru.yandex.chemodan.app.djfs.core.album.FetchAestheticsForMauFromIndexerTask;
import ru.yandex.chemodan.app.djfs.core.album.FetchCoordinatesFromIndexerTask;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumResourceProcessingTask;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsAlbumsTaskQueueName;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class YtFetchCoordinatesSubmitTask extends YcridOnetimeTaskSupport<YtFetchCoordinatesSubmitTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(GeoAlbumResourceProcessingTask.class);

    private final YtHelper ytHelper;
    private final BazingaTaskManager bazingaTaskManager;

    public YtFetchCoordinatesSubmitTask(YtHelper ytHelper, BazingaTaskManager bazingaTaskManager)
    {
        super(Parameters.class);
        this.ytHelper = ytHelper;
        this.bazingaTaskManager = bazingaTaskManager;
    }

    public YtFetchCoordinatesSubmitTask(String ytPath, int lowerIndex, int upperIndex)
    {
        super(new Parameters(ytPath, lowerIndex, upperIndex));
        this.ytHelper = null;
        this.bazingaTaskManager = null;
    }

    public YtFetchCoordinatesSubmitTask(String ytPath, int lowerIndex, int upperIndex,
            boolean isTemporaryAestheticsSave)
    {
        super(new Parameters(ytPath, lowerIndex, upperIndex, isTemporaryAestheticsSave));
        this.ytHelper = null;
        this.bazingaTaskManager = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context)
    {
        YPath pathWithRange = YPath.simple(parameters.ytPath).withRange(parameters.lowerIndex, parameters.upperIndex);

        ytHelper.tables().read(pathWithRange, YTableEntryTypes.JACKSON, row -> {
            JsonNode node = row.path("uid");
            String uid = node.asText();

            if (uid.isEmpty()) {
                return;
            }

            if (parameters.isTemporaryAestheticsSave) {
                bazingaTaskManager.schedule(
                        new FetchAestheticsForMauFromIndexerTask(uid)
                );
            } else {
                bazingaTaskManager.schedule(
                        new FetchCoordinatesFromIndexerTask(uid)
                );
            }
        });
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsAlbumsTaskQueueName.ALBUMS_INDEXER_TASKS;
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String ytPath;
        private final int lowerIndex;
        private final int upperIndex;
        private final boolean isTemporaryAestheticsSave;

        Parameters(String ytPath, int lowerIndex, int upperIndex) {
            this(ytPath, lowerIndex, upperIndex, false);
        }

        Parameters(String ytPath, int lowerIndex, int upperIndex, boolean isTemporaryAestheticsSave) {
            this.ytPath = ytPath;
            this.lowerIndex = lowerIndex;
            this.upperIndex = upperIndex;
            this.isTemporaryAestheticsSave = isTemporaryAestheticsSave;
        }
    }
}
