package ru.yandex.chemodan.app.djfs.api;

import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.legacy.web.LegacyMpfsExceptionHandler;
import ru.yandex.chemodan.app.djfs.core.user.CheckBlockedInterceptor;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.web.ConnectionIdActionDispatcherInterceptor;
import ru.yandex.chemodan.app.djfs.core.web.DjfsTskvLog4jRequestLog;
import ru.yandex.chemodan.app.djfs.core.web.DynamicMasterSlavePolicyInterceptor;
import ru.yandex.chemodan.app.djfs.core.web.JsonStringResultSerializer;
import ru.yandex.chemodan.app.djfs.core.web.LoggingAnyExceptionHandler;
import ru.yandex.chemodan.app.djfs.core.web.MasterSlavePolicyOverridesRegistry;
import ru.yandex.chemodan.app.djfs.core.web.UidActionDispatcherInterceptor;
import ru.yandex.chemodan.util.web.A3JettyConfiguration;
import ru.yandex.chemodan.util.web.MasterSlaveUnavailableExceptionHandler;
import ru.yandex.chemodan.util.web.SuppressInvocationInfoJsonPojoResultSerializer;
import ru.yandex.chemodan.util.web.interceptors.ThreadLocalCacheInterceptor;
import ru.yandex.commune.a3.ActionApp;
import ru.yandex.commune.a3.ActionConfigurationBuilder;
import ru.yandex.commune.a3.ActionConfigurator;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.parameter.bind.BenderJsonListParameterBinder;
import ru.yandex.commune.a3.action.parameter.bind.BenderParameterBinder;
import ru.yandex.commune.a3.action.parameter.convert.DefaultConverters;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.commune.a3.action.result.ResponseInterceptor;
import ru.yandex.commune.a3.action.result.SimpleResultSerializer;
import ru.yandex.commune.a3.action.result.error.A3ExceptionHandler;
import ru.yandex.commune.a3.action.result.error.HttpStatusCodeSourceExceptionHandler;
import ru.yandex.commune.a3.action.result.error.nested.ExceptionWithRootElementHandler;
import ru.yandex.commune.a3.action.result.type.MediaType;
import ru.yandex.commune.a3.security.SecurityExceptionHandler;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.ReadableInstantConfigurableMarshaller;
import ru.yandex.misc.spring.ApplicationContextUtils;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;

public class DjfsApiBaseContextConfiguration {
    @Bean
    public A3JettyConfiguration a3JettyServletsConfiguration() {
        return new A3JettyConfiguration(Tuple2List.fromPairs("/api/*", "")) {
            @Override
            public void postConstruct(SingleWarJetty jetty, ActionApp actionApp) {
                jetty.setRequestLogFactory(DjfsTskvLog4jRequestLog::new);
            }
        };
    }

    @Bean
    public ActionApp actionApp(ApplicationContext context, ApplicationInfo applicationInfo, UserDao userDao,
            MasterSlavePolicyOverridesRegistry masterSlavePolicyOverridesRegistry)
    {
        BenderMapper mapper = new BenderMapper(BenderConfiguration.cons(
                MembersToBind.WITH_ANNOTATIONS, false,
                CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(Instant.class, new ReadableInstantConfigurableMarshaller(ISODateTimeFormat.dateTime()))
                        .build()
        ));

        ListF<CloneableAction> cloneableActions = ApplicationContextUtils.beansOfType(context, CloneableAction.class);

        ListF<Object> actionContainers =
                ApplicationContextUtils.beansWithAnnotationList(context, ActionContainer.class).get2();

        ActionConfigurationBuilder builder = ActionConfigurationBuilder
                .cons(applicationInfo)
                .setConverters(DefaultConverters.all().getConverters())
                .setExceptionHandlers(Cf.list(
                        context.getBean(LegacyMpfsExceptionHandler.class),
                        new MasterSlaveUnavailableExceptionHandler(),
                        new HttpStatusCodeSourceExceptionHandler(),
                        new ExceptionWithRootElementHandler(),
                        new SecurityExceptionHandler(),
                        new A3ExceptionHandler(),
                        new LoggingAnyExceptionHandler()
                ))
                .setResultSerializers(Cf.list(
                        new SimpleResultSerializer(),
                        new SuppressInvocationInfoJsonPojoResultSerializer(mapper),
                        new JsonStringResultSerializer()
                ))
                .setDispatcherInterceptors(Cf.list(
                        new UidActionDispatcherInterceptor(),
                        new ConnectionIdActionDispatcherInterceptor()
                ))
                .setInvocationInterceptors(Cf.list(
                        new ThreadLocalCacheInterceptor(),
                        new ResponseInterceptor(),
                        new CheckBlockedInterceptor(userDao),
                        new DynamicMasterSlavePolicyInterceptor(masterSlavePolicyOverridesRegistry)
                ))
                .setParameterBinders(Cf.list(
                        new BenderParameterBinder(mapper),
                        new BenderJsonListParameterBinder(mapper)
                ))
                .setResultType(MediaType.APPLICATION_JSON);

        return ActionConfigurator.configure(actionContainers.plus(cloneableActions), builder);
    }

    @Bean
    public MasterSlavePolicyOverridesRegistry masterSlavePolicyOverridesRegistry(@Qualifier("zkRoot") ZkPath zkRoot,
            ZkManager zkManager)
    {
        MasterSlavePolicyOverridesRegistry registry =
                new MasterSlavePolicyOverridesRegistry(zkRoot.child("master-slave-policy-overrides"));
        zkManager.addClient(registry);
        return registry;
    }
}
