package ru.yandex.chemodan.app.djfs.core;

import java.net.URI;
import java.util.function.Function;
import java.util.function.Supplier;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.util.encrypt.HmacDigestUtil;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.http.UriBuilder;

public class DownloadUrlGenerator {
    private Duration defaultExpirationPeriod;
    private String hmacTokenPassword;
    private LegacyMpfsAes aes;
    private String baseUrl;

    public DownloadUrlGenerator(String baseUrl, String hmacTokenPassword, Duration defaultExpirationPeriod,
            LegacyMpfsAes aes)
    {
        this.baseUrl = baseUrl;
        this.hmacTokenPassword = hmacTokenPassword;
        this.defaultExpirationPeriod = defaultExpirationPeriod;
        this.aes = aes;
    }

    public URI makePublicPreviewUrl(Supplier<DjfsUid> ownerUidProvider, String previewStid, Option<String> mimetypeO, String filename,
            Option<Dimension> size, boolean isInline, boolean isEternal)
    {
        return preparePreviewUrl(DjfsUid.cons("0"),
                ownerUidProvider, previewStid, mimetypeO, filename, size.map(Dimension::toString),
                Option.empty(), isInline, isEternal).build();
    }

    public URI makeUserPreviewUrl(
            DjfsUid uid, Supplier<DjfsUid> ownerUidProvider, String previewStid, Option<String> mimetypeO, String filename,
            Option<Dimension> size, boolean isInline, boolean isEternal)
    {
        return preparePreviewUrl(uid, ownerUidProvider, previewStid, mimetypeO, filename, size.map(Dimension::toString),
                Option.empty(), isInline, isEternal).build();
    }

    public UriBuilder prepareUserPreviewUrl(
            DjfsUid uid, Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, Option<String> size, Option<String> crop,
            boolean isInline, boolean isEternal)
    {
        return preparePreviewUrl(uid, ownerUidProvider, file.getPreviewStid().get(), file.getMimetype(),
                file.getDisplayName(), size, crop, isInline, isEternal);
    }

    public UriBuilder preparePreviewUrl(
            DjfsUid uid, Supplier<DjfsUid> ownerUidProvider, String previewStid, Option<String> mimetypeO, String filename,
            Option<String> size, Option<String> crop, boolean isInline, boolean isEternal)
    {
        String previewMimetype = getPreviewMimetype(mimetypeO);
        UrlData urlData = getUrlData(uid, ownerUidProvider);
        UriBuilder builder = baseUrlBuilder(
                "preview", urlData.getUidValue(), urlData.getOwnerUid(), previewStid, Option.of(previewMimetype), filename, isInline, isEternal);

        if (size.isPresent()) {
            builder.addParam("size", size.get());
        }

        if (crop.isPresent()) {
            builder.addParam("crop", crop.get());
        }

        return builder;
    }

    public URI makeDownloadUrl(
            DjfsUid uid, Supplier<DjfsUid> ownerUidProvider, String stid, Option<String> mimetypeO, String filename, boolean isInline,
            boolean isEternal)
    {
        UrlData urlData = getUrlData(uid, ownerUidProvider);
        return baseUrlBuilder(urlData.getScope(), urlData.getUidValue(), urlData.getOwnerUid(), stid, mimetypeO, filename, isInline, isEternal).build();
    }

    public UriBuilder prepareDownloadUrl(
            DjfsUid uid, Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, boolean isInline, boolean isEternal,
            Function<? super FileDjfsResource, String> fileStidProvider,
            Function<FileDjfsResource, Option<String>> mimetypeProvider, SetF<DownloadUrlAdditionalField> additionalFields)
    {
        UrlData urlData = getUrlData(uid, ownerUidProvider);
        UriBuilder builder = baseUrlBuilder(
                urlData.getScope(),
                urlData.getUidValue(),
                urlData.getOwnerUid(),
                fileStidProvider.apply(file),
                mimetypeProvider.apply(file),
                file.getDisplayName(),
                isInline,
                isEternal);
        additionalFields.forEach(field -> field.getValue(file).ifPresent(value -> builder.addParam(field.getName(), value)));
        return builder;
    }

    private UrlData getUrlData(DjfsUid uid, Supplier<DjfsUid> ownerUidProvider) {
        if (uid.isShared()) {
            return new UrlData("share", "0", "0");
        }
        return new UrlData("disk", uid.asString(), ownerUidProvider.get().asString());
    }

    private UriBuilder baseUrlBuilder(
            String scope, String uid, String ownerUid, String previewStid, Option<String> mimetypeO,
            String filename, boolean isInline, boolean isEternal)
    {
        String hash = "";
        int limit = 0;

        String mimetype = mimetypeO.getOrElse("application/octet-stream");

        String timestampValue = isEternal ? "inf" : Integer
                .toHexString(InstantUtils.toSeconds(Instant.now().plus(defaultExpirationPeriod)));
        String disposition = isInline ? "inline" : "attachment";

        String token = generateSignedToken(Cf.arrayList(
                timestampValue,
                previewStid,
                uid,
                filename,
                mimetype,
                disposition,
                hash,
                Integer.toString(limit)
        ));

        String encryptedStid = aes.encrypt(previewStid);

        return UriBuilder.cons(baseUrl)
                .appendPath(scope)
                .appendPath(token)
                .appendPath(timestampValue)
                .appendPath(encryptedStid)
                .addParam("uid", uid)
                .addParam("filename", filename)
                .addParam("disposition", disposition)
                .addParam("hash", hash)
                .addParam("limit", Integer.toString(limit))
                .addParam("content_type", mimetype)
                .addParam("tknv", "v2")
                .addParam("owner_uid", ownerUid);
    }

    private String generateSignedToken(ListF<String> params) {
        String message = String.join("-", params);
        return Hex.encode(HmacDigestUtil.sign(message, hmacTokenPassword));
    }

    private String getPreviewMimetype(Option<String> mimetypeO) {
        final String defaultPreviewMimetype = "image/jpeg";

        if (!mimetypeO.isPresent()) {
            return defaultPreviewMimetype;
        }
        String mimetype = mimetypeO.get();
        if (mimetype.equals("image/gif") || mimetype.equals("image/png") || mimetype.equals("image/x-png")) {
            return mimetype;
        } else {
            return defaultPreviewMimetype;
        }
    }

    @Data
    private static class UrlData {

        private final String scope;

        private final String uidValue;

        private final String ownerUid;
    }
}
