package ru.yandex.chemodan.app.djfs.core;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

/**
 * Код обработки стандартных методов отсюда: https://docs.oracle.com/javase/8/docs/technotes/guides/reflection/proxy.html
 *
 * @author eoshch
 */
public abstract class ProxyInvocationHandlerBase implements InvocationHandler {
    private static Method hashCodeMethod;
    private static Method equalsMethod;
    private static Method toStringMethod;

    static {
        try {
            hashCodeMethod = Object.class.getMethod("hashCode", (Class<?>[]) null);
            equalsMethod = Object.class.getMethod("equals", Object.class);
            toStringMethod = Object.class.getMethod("toString", (Class<?>[]) null);
        } catch (NoSuchMethodException e) {
            throw new NoSuchMethodError(e.getMessage());
        }
    }

    public final Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        Class declaringClass = method.getDeclaringClass();

        if (declaringClass == Object.class) {
            if (method.equals(hashCodeMethod)) {
                return proxyHashCode(proxy);
            } else if (method.equals(equalsMethod)) {
                return proxyEquals(proxy, args[0]);
            } else if (method.equals(toStringMethod)) {
                return proxyToString(proxy);
            } else {
                throw new InternalError("unexpected Object method dispatched: " + method);
            }
        }

        return invokeImpl(proxy, method, args);
    }

    protected abstract Object invokeImpl(Object proxy, Method method, Object[] args) throws Throwable;

    private Integer proxyHashCode(Object proxy) {
        return System.identityHashCode(proxy);
    }

    private Boolean proxyEquals(Object proxy, Object other) {
        return (proxy == other ? Boolean.TRUE : Boolean.FALSE);
    }

    private String proxyToString(Object proxy) {
        return proxy.getClass().getName() + '@' + Integer.toHexString(proxy.hashCode());
    }

}
