package ru.yandex.chemodan.app.djfs.core.album;

import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumAppendActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumItemRemoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumappend.AlbumAppendCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumitemremove.AlbumItemRemoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public abstract class AbstractAlbumManager extends AlbumUpdateManager {
    private static final Logger logger = LoggerFactory.getLogger(AbstractAlbumManager.class);
    private final ShareInfoManager shareInfoManager;
    protected final DjfsResourceDao djfsResourceDao;

    protected AbstractAlbumManager(
            AlbumDao albumDao,
            AlbumItemDao albumItemDao,
            AlbumDeltaDao albumDeltaDao,
            TransactionUtils transactionUtils,
            XivaPushGenerator xivaPushGenerator,
            ShareInfoManager shareInfoManager,
            DjfsResourceDao djfsResourceDao
    ) {
        super(albumDao, albumDeltaDao, albumItemDao, transactionUtils, xivaPushGenerator);
        this.shareInfoManager = shareInfoManager;
        this.djfsResourceDao = djfsResourceDao;
    }

    abstract public AlbumType getAlbumType();

    public ListF<AlbumSetAttrUpdater> setTitle(Option<String> titleO)
    {
        return titleO.map(title -> new AlbumSetAttrUpdater(
                AlbumUtils.generateSetTitleChangeFields(title),
                AlbumParam.title(title)));
    }

    public ListF<AlbumSetAttrUpdater> setLayout(Option<String> rawLayoutO)
    {
        return rawLayoutO.map(AlbumLayoutType.R::fromValue)
                .map(layout -> new AlbumSetAttrUpdater(
                AlbumUtils.generateSetLayoutChangeFields(layout),
                AlbumParam.layout(layout)));
    }

    protected ListF<AlbumSetAttrUpdater> setCover(
            DjfsUid uid, Album album, Option<String> rawCover, Option<Double> coverOffsetY
    )
    {
        ListF<AlbumSetAttrUpdater> result = Cf.arrayList();
        if (rawCover.isPresent()) {
            Option<AlbumItem> coverO = Option.empty();
            try {
                //  значит указали номер элемента
                int coverAlbumItemNum = Integer.parseInt(rawCover.get());
                ListF<AlbumItem> allItems = albumItemDao.getExistingAlbumItems(uid, album.getId());
                coverO = allItems.getO(coverAlbumItemNum);
                if (!coverO.isPresent()) {
                    coverO = allItems.firstO();
                }
            } catch (NumberFormatException nfe) {
                //  значит указали идентификатор элемента
                coverO = albumItemDao.findByItemId(uid, rawCover.get());
            }
            DjfsFileId newSelectedCover = DjfsFileId.cons(coverO.get().getObjectId());
            Tuple2<ObjectId, DjfsResourceId> newCover = Tuple2.tuple(coverO.get().getId(), DjfsResourceId.cons(uid, newSelectedCover));

            result.add(new AlbumSetAttrUpdater(
                    AlbumUtils.generateSetCoverChangeFields(newCover._1, newCover._2),
                    AlbumParam.cover(newCover._1)
            ));

            result.add(new AlbumSetAttrUpdater(
                    Cf.list(), // no special deltas, clients don't need to know
                    AlbumParam.coverAuto(false) // we are here only for manual change
            ));

            if (!coverOffsetY.isPresent()) {
                result.add(new AlbumSetAttrUpdater(
                        AlbumUtils.generateSetCoverOffsetYChangeFields(0d),
                        AlbumParam.coverOffsetY(0d)
                ));
            }
        }
        return result;
    }

    public ListF<AlbumSetAttrUpdater> setDescription(Option<String> descriptionO)
    {
        return descriptionO.map(description -> new AlbumSetAttrUpdater(
                AlbumUtils.generateSetDescriptionChangeFields(description),
                AlbumParam.description(description)));
    }

    public ListF<AlbumSetAttrUpdater> setCoverOffset(Option<Double> coverOffsetYO)
    {
        return coverOffsetYO.map(coverOffsetY -> new AlbumSetAttrUpdater(
                AlbumUtils.generateSetCoverOffsetYChangeFields(coverOffsetY),
                AlbumParam.coverOffsetY(coverOffsetY)));
    }

    public ListF<AlbumSetAttrUpdater> setDateModified()
    {
        ListF<AlbumSetAttrUpdater> result = Cf.arrayList();
        Instant now = Instant.now();
        result.add(new AlbumSetAttrUpdater(
                AlbumUtils.generateSetDateModifyChangeFields(now),
                AlbumParam.dateModified(now)
        ));
        return result;
    }

    public ListF<AlbumSetAttrUpdater> setFlags(Option<ListF<String>> flagsO)
    {
        return flagsO.map(flags -> new AlbumSetAttrUpdater(
                Cf.list(),
                AlbumParam.flags(flags)));
    }

    public ListF<AlbumSetAttrUpdater> setFotkiAlbumId(Option<Long> fotkiAlbumIdO)
    {
        return fotkiAlbumIdO.map(fotkiAlbumId -> new AlbumSetAttrUpdater(
                Cf.list(),
                AlbumParam.fotkiAlbumId(fotkiAlbumId)));
    }

    public void saveDeltasAndUpdateAlbumDB(Album album, long currentRevision, ListF<AlbumSetAttrUpdater> updaters) {
        ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
        ListF<AlbumParam> albumUpdateDB = Cf.arrayList();
        for (AlbumSetAttrUpdater updater: updaters) {
            albumUpdateDB.add(updater.getAlbumParam());
            albumUpdateChanges.addAll(updater.getDeltas());
        }
        albumDao.setParams(album, albumUpdateDB);
        AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);
        albumDeltaDao.insert(updateAlbumDelta);
    }

    public Option<ListF<String>> getUniqFlagsList(Option<String> rawFlags) {
        Option<ListF<String>> flags = Option.empty();
        if (rawFlags.isPresent()) {
            ListF<String> flagsList = Cf.arrayList();
            for (String flag: rawFlags.get().split(",")) {
                flagsList.add(flag.trim());
            }
            flags = Option.of((ListF<String>) flagsList.unique());
        }
        return flags;
    }

    public MapF<ObjectId, Album> getAlbumsWithTypeCheck(DjfsUid uid, ListF<ObjectId> ids) {
        MapF<ObjectId, Album> albums = getAlbumsWithTypeCheckSkipMissing(uid, ids);
        ids.unique().minus(albums.keySet()).forEach(missingId -> {
            throw new AlbumsNotFoundException("No album found for id " + missingId);
        });
        return albums;
    }

    public MapF<ObjectId, Album> getAlbumsWithTypeCheckSkipMissing(DjfsUid uid, ListF<ObjectId> ids) {
        AlbumType expectedType = getAlbumType();
        MapF<ObjectId, Album> albums = albumDao.findAlbums(uid, ids).toMap(Album::getId, album -> album);
        albums.values().forEach(album -> {
            AlbumType type = album.getType();
            if (type != expectedType) {
                throw new AlbumTypeMismatchException(album.getId(), expectedType, type);
            }
        });
        return albums;
    }

    public ListF<Tuple2<AlbumItem, FileDjfsResource>> addPhotosToAlbum(
            ListF<DjfsResource> resources,
            Album album,
            boolean skipExisting,
            AlbumAppendCallbacks callbacks,
            boolean allowNoAction,
            boolean autoHandle
    ) {
        return addPhotosToAlbum(
                resources.zipWith(x -> Option.empty()),
                album,
                skipExisting,
                callbacks,
                allowNoAction,
                autoHandle);
    }

    public ListF<Tuple2<AlbumItem, FileDjfsResource>> addPhotosToAlbum(
            Tuple2List<DjfsResource, Option<FaceInfo>> resourcesInfos,
            Album album,
            boolean skipExisting,
            AlbumAppendCallbacks callbacks,
            boolean allowNoAction,
            boolean autoHandle
    ) {
        final DjfsUid uid = album.getUid();
        final ListF<DjfsResource> resources = resourcesInfos.get1();
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);

        for (DjfsResource resource: resources) {
            if (!(resource instanceof FileDjfsResource)){
                throw new AlbumUnableToAppendException("Resource is not file: " + resource.getPath().getFullPath());
            }
        }
        ListF<Tuple2<AlbumItem, FileDjfsResource>> items = Cf.arrayList();
        if (resources.isEmpty()) {
            return items;
        }
        Option<Long> revisionIfModified = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            ListF<String> existingItem = albumItemDao.findObjectsInAlbum(uid, album.getId(),
                    resources.map(x -> x.getFileId().get().getValue())).map(AlbumItem::getObjectId);

            ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
            ListF<Tuple2<FileDjfsResource, Option<FaceInfo>>> filesInfos =
                    resourcesInfos
                            .map1(x -> (FileDjfsResource)x)
                            .filterBy1(x ->
                                    x.getResourceId().isPresent()
                                            && (!skipExisting
                                            || !existingItem.containsTs(x.getResourceId().get().getFileId().getValue()))
                            );
            if (filesInfos.isEmpty()) {
                logger.info("Items are already in the album " + album.getId().toHexString() + ", adding is skipped");
                if (!allowNoAction) {
                    throw new AlbumUnableToAppendException(
                            "Items are already in the album " + album.getId().toHexString() + ", adding is skipped"
                    );
                } else {
                    return Option.empty();
                }
            }
            for (Tuple2<FileDjfsResource, Option<FaceInfo>> resourceInfo: filesInfos) {
                FileDjfsResource resource = resourceInfo.get1();
                DjfsResourceId fileResourceId = resource.getResourceId().get();
                AlbumItem item = AlbumItem.cons(album, fileResourceId.getFileId(), itemOrderIndex(resource), resourceInfo.get2());
                albumItemDao.insert(item);
                items.add(Tuple2.tuple(item, resource));
                currentRevision += 1;
                // TODO Тут хорошо бы завести дельту на балковое добавление? и менять ревизию не на 1, но не в этот раз
                AlbumDelta albumItemDelta = AlbumUtils.createInsertAlbumItemDelta(album, item, currentRevision);
                albumDeltaDao.insert(albumItemDelta);
                if (album.getCoverId().isPresent()) {
                    if (album.isCoverAuto()) {
                        changeCoverIfNeeded(album, item);
                    }
                } else {
                    albumDao.setCover(album, item.getId());
                    albumUpdateChanges.addAll(AlbumUtils.generateSetCoverChangeFields(item.getId(), fileResourceId));
                }
                // TODO don't update shareInfo for faces
                Option<ShareInfo> shareInfo = shareInfoManager.get(resource);
                AlbumAppendActivity albumAppendActivity = new AlbumAppendActivity(principal, album, resource, item,
                        shareInfo);
                callbacks.callAfterAlbumAppendOutsideTransaction(albumAppendActivity);
            }
            int itemsCountAfter = albumItemDao.countObjectsInAlbum(uid, album.getId());
            albumUpdateChanges.addAll(AlbumUtils.generateSetItemsCountChangeFields(itemsCountAfter));

            currentRevision += 1;
            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);
            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Option.of(currentRevision);
        });
        revisionIfModified.ifPresent(revision -> {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, revision);
            for (Tuple2<AlbumItem, FileDjfsResource> item : items) {
                xivaPushGenerator.sendAlbumItemAppendPush(uid, album, item._1, autoHandle);
            }
        });
        return items;
    }

    protected void changeCoverIfNeeded(Album album, AlbumItem item) {
        /* do nothing by default */
    }

    protected abstract Option<Double> itemOrderIndex(FileDjfsResource resource);

    public void removePhotoFromAlbum(DjfsUid uid, Album album, String itemId, AlbumItemRemoveCallbacks callbacks,
                                     boolean autoHandle) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);
        Option<AlbumItem> itemO = albumItemDao.findByItemId(uid, itemId);
        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            Option<AlbumItem> item = albumItemDao.findByItemId(uid, itemId);
            if (!item.isPresent()) {
                logger.info("There is no item " + itemId + " in album " + album.getId().toString());
                return Tuple2.tuple(false, 0L);
            }
            AlbumItem albumItem = item.get();
            Option<ShareInfo> shareInfo = Option.empty();
            DjfsUid ownerUid = uid;
            if (albumItem.getGroupId().isPresent()) {
                shareInfo = shareInfoManager.get(albumItem.getGroupId().get());
                ownerUid = shareInfo.get().getOwnerUid();
            }
            albumItemDao.removeFromAlbum(uid, album.getId(), albumItem.getObjectId());
            DjfsResourceId resourceId = DjfsResourceId.cons(ownerUid, albumItem.getObjectId());
            ListF<DjfsResource> resources = djfsResourceDao.find(resourceId);
            if (!resources.isEmpty()) {
                DjfsResource resource = resources.first();

                AlbumItemRemoveActivity albumItemRemoveActivity = new AlbumItemRemoveActivity(
                        principal, album, resource, albumItem, shareInfo
                );
                callbacks.callAfterAlbumItemRemoveOutsideTransaction(albumItemRemoveActivity);
            }

            currentRevision += 1;
            AlbumDelta albumItemDelta = AlbumUtils.createDeleteAlbumItemDelta(album, albumItem, currentRevision);
            albumDeltaDao.insert(albumItemDelta);

            ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
            int itemsCount = albumItemDao.countObjectsInAlbum(uid, album.getId());
            albumUpdateChanges.addAll(AlbumUtils.generateSetItemsCountChangeFields(itemsCount));

            if (album.getCoverId().isPresent() && albumItem.getId().equals(album.getCoverId().get())) {
                albumDao.setCoverAuto(album, true);
                ListF<AlbumDeltaFieldChange> updateCoverDelta = AlbumUtils.selectNewCover(album, albumItemDao, albumDao);
                albumUpdateChanges.addAll(updateCoverDelta);
            }

            currentRevision += 1;
            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            if (itemO.isPresent()) {
                xivaPushGenerator.sendAlbumItemRemovedPush(uid, album, itemO.get(), autoHandle);
            }
        }
    }
}
