package ru.yandex.chemodan.app.djfs.core.album;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;
import org.bson.types.ObjectId;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumAppendPojo;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumDeltaItemsPojo;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumDeltasPojo;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumItemPojo;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumPojo;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumSnapshotPojo;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.DjfsNotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourcePathException;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumappend.AlbumAppendCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumitemremove.AlbumItemRemoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumremove.AlbumRemoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumsetattr.AlbumSetAttrCallbacks;
import ru.yandex.chemodan.app.djfs.core.history.EventHistoryLogger;
import ru.yandex.chemodan.app.djfs.core.legacy.AsyncOperationResultPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.PreviewFormattingOptions;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.FormattingContext;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.ReadResourceEndpoint;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.ResourcePojo;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.ResourcePojoBuilder;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.UserPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.web.TranslateExceptionToLegacy;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.publication.PublicationManager;
import ru.yandex.chemodan.app.djfs.core.share.Group;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.CheckBlocked;
import ru.yandex.chemodan.app.djfs.core.user.ClientInputDataProcessor;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.chemodan.util.web.interceptors.WithThreadLocalCache;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundJsonListByBender;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;


@ActionContainer
@RequiredArgsConstructor
public class AlbumActions {
    private static final String FACES_REINDEXED_PATH = "/v1/albums/faces/indexer/reindexed";

    private static final Logger logger = LoggerFactory.getLogger(AlbumActions.class);

    private final UserDao userDao;
    private final AlbumDao albumDao;
    private final AlbumItemDao albumItemDao;
    private final AlbumDeltaDao albumDeltaDao;
    private final GeoAlbumManager geoAlbumManager;
    private final PersonalAlbumManager personalAlbumManager;
    private final FacesAlbumManager facesAlbumManager;
    private final FavoritesAlbumManager favoritesAlbumManager;
    private final Filesystem filesystem;
    private final BazingaTaskManager bazingaTaskManager;
    private final GeoAlbumGenerationProperties properties;
    private final ShareInfoManager shareInfoManager;
    private final ClientInputDataProcessor clientInputDataProcessor;
    private final EventHistoryLogger eventHistoryLogger;
    private final PublicationManager publicationManager;
    private final ResourcePojoBuilder resourcePojoBuilder;
    private final DjfsResourceDao djfsResourceDao;
    private final AlbumMergeManager albumMergeManager;
    private final MpfsClient mpfsClient;

    private static final int DELTAS_MAX_LIMIT = 40000;
    private static final int DELTAS_LIMIT = 100;

    @CheckBlocked
    @WithThreadLocalCache
    @Path(value = "/v1/albums/snapshot", methods = {HttpMethod.GET})
    public AlbumSnapshotPojo snapshot(
            @RequestParam("uid") String rawUid,
            @RequestListParam("type") ListF<AlbumType> albumTypes,
            @RequestListParam("collectionIds") ListF<String> collectionIds
    )
    {
        TlCache.flush();

        DjfsUid uid = DjfsUid.cons(rawUid);

        if (collectionIds.length() > 1 && collectionIds.containsTs(CollectionId.ALBUM_LIST_COLLECTION_ID)) {
            throw new InvalidCollectionIdException(
                    CollectionId.ALBUM_LIST_COLLECTION_ID + " collection id should not be mixed with album collections"
            );
        }

        Option<Long> currentRevisionO = albumDeltaDao.getCurrentRevisionWithoutLock(uid);
        if (albumTypes.containsTs(AlbumType.GEO)) {
            if (!currentRevisionO.isPresent()) {
                if (properties.getGeoAlbumCatchupGenerationEnabled().get()) {
                    geoAlbumManager.startAlbumCreationProcess(uid);
                }
                throw new AlbumsNotFoundException("No albums were created for user: " + uid.asString());
            }
        }

        if (collectionIds.containsTs(CollectionId.ALBUM_LIST_COLLECTION_ID)) {
            ListF<ExtendedAlbum> albumsWithMeta = albumDao.getExtendedAlbums(uid, albumTypes);

            if (albumTypes.containsTs(AlbumType.PERSONAL)) {
                albumsWithMeta = albumsWithMeta.filterNot(x -> x.album.getFotkiAlbumId().isPresent());
            }

            if (albumsWithMeta.isEmpty()) {
                throw new AlbumsNotFoundException("User has no albums: " + uid.asString());
            }
            return AlbumSnapshotPojo.consForAlbums(albumsWithMeta, currentRevisionO);
        } else {
            ListF<ObjectId> albumIds = collectionIds.map(CollectionId::parseAlbumCollectionId);
            ListF<AlbumItem> albumItems = albumItemDao.getAllAlbumItems(uid, albumIds, AlbumUtils.GEO_ALBUM_SIZE_LIMIT);
            ListF<Group> joinedGroups = shareInfoManager.getJoinedGroups(uid);

            return AlbumSnapshotPojo.consForItems(albumItems,
                    joinedGroups.toTuple2List(g -> Tuple2.tuple(g.getId(), g.getUid())));
        }
    }

    @WithThreadLocalCache
    @Path(value = "/v1/albums/deltas", methods = {HttpMethod.GET})
    public AlbumDeltasPojo deltas(
            @RequestParam("uid") String rawUid,
            @RequestParam("baseRevision") long baseRevision,
            @RequestParam("limit") Option<Integer> limitO,
            @RequestListParam(value = "type", required = false) ListF<AlbumType> albumTypesList
    )
    {
        TlCache.flush();

        SetF<AlbumType> albumTypes = albumTypesList.isEmpty() ? Cf.set(AlbumType.GEO) : albumTypesList.unique();

        DjfsUid uid = DjfsUid.cons(rawUid);
        int limit = limitO.filter(x -> x <= DELTAS_LIMIT && x > 0).getOrElse(DELTAS_LIMIT);

        Option<Long> currentRevisionO = albumDeltaDao.getCurrentRevisionWithoutLock(uid);
        if (!currentRevisionO.isPresent()) {
            throw new AlbumsNotFoundException("No albums were created for user " + uid.asString());
        }
        long currentRevision = currentRevisionO.get();

        if (currentRevision == baseRevision) {
            return new AlbumDeltasPojo(baseRevision, currentRevisionO.get(), limit, 0, Cf.list());
        }

        Option<AlbumDeltaRaw> baseDelta = albumDeltaDao.findRaw(uid, baseRevision + 1);
        if (!baseDelta.isPresent()) {
            throw new RevisionNotFoundException("Delta with revision " + baseRevision + " not found");
        }

        int deltasTotalCount = albumDeltaDao.count(uid, baseRevision, currentRevision);

        if (deltasTotalCount != currentRevision - baseRevision) {
            throw new RevisionNotFoundException("Some deltas missing between "
                    + baseRevision + " and " + currentRevision + ". " + deltasTotalCount + " deltas found");
        }

        if (deltasTotalCount >= DELTAS_MAX_LIMIT) {
            throw new RevisionIsTooOldException("Revision " + baseRevision + " is too old for user " + uid.asString());
        }

        ListF<AlbumDeltaRaw> albumDeltas = albumDeltaDao.findRaw(uid, baseRevision, currentRevision, limit);
        ListF<AlbumDeltaItemsPojo> items = albumDeltas.map(
                x -> {
                    if (albumTypes.containsTs(x.getAlbumType().getOrElse(AlbumType.GEO)))
                        return new AlbumDeltaItemsPojo(x.getId(),
                                RawJsonValue.defaultRawJsonValue(x.getChanges()),
                                x.revision - 1, x.revision);
                    else
                        return new AlbumDeltaItemsPojo(x.getId(),
                                RawJsonValue.emptyRawValueFromString(),
                                x.revision - 1, x.revision);
                }
        );
        return new AlbumDeltasPojo(baseRevision, currentRevision, limit, deltasTotalCount, items);
    }

    @SynchronizedToAlbumLocked
    @WithThreadLocalCache
    @Path(value = "/v1/albums/exclude_photo_from_geo_album", methods = {HttpMethod.POST})
    public void excludeItemFromGeoAlbum(
            @RequestParam("uid") String rawUid,
            @RequestParam("path") String rawPath
    )
    {
        TlCache.flush();

        DjfsUid uid = DjfsUid.cons(rawUid);
        DjfsResourcePath path = DjfsResourcePath.cons(uid, rawPath);

        Option<DjfsResource> resource = filesystem.find(DjfsPrincipal.cons(uid), path, Option.of(ReadPreference.primary()));
        if (!resource.isPresent() || !(resource.get() instanceof FileDjfsResource)) {
            throw new AlbumItemInvalidPathException(path.getPath());
        }
        FileDjfsResource file = (FileDjfsResource) resource.get();

        ListF<DefaultCityStrategy> removeStrategies = geoAlbumManager.getRemoveStrategies();
        CollectionF<Integer> geoIds = geoAlbumManager.getAllResourceGeoIds(removeStrategies, file).values();
        if (geoIds.isEmpty()) {
            throw new AlbumItemGeoNotFoundException(path.getPath());
        }
        geoIds.forEach(i -> geoAlbumManager.removeFromGeoAlbum(file, i));
    }

    @WithThreadLocalCache
    @Path(value = "/v1/albums/submit_coordinates_from_search_task", methods = {HttpMethod.POST})
    public void excludeItemFromGeoAlbum(
            @RequestParam("uid") String rawUid)
    {
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid);

        bazingaTaskManager.schedule(
                new FetchCoordinatesFromIndexerTask(uid.asString())
        );
    }

    @Path(value = "/v1/albums/faces/reindex", methods = {HttpMethod.POST})
    public OkPojo startIndexFaces(
            @RequestParam("uid") String rawUid, @RequestParam("resetAlbums") Option<Boolean> resetAlbumsO
    )
    {
        final DjfsUid uid = DjfsUid.cons(rawUid);
        final boolean resetAlbums = resetAlbumsO.getOrElse(false);
        facesAlbumManager.startIndexing(uid, resetAlbums);
        return new OkPojo();
    }

    public static String getFacesReindexedQuery(DjfsUid uid) {
        return FACES_REINDEXED_PATH + "?uid=" + uid;
    }

    @Path(value = FACES_REINDEXED_PATH, methods = {HttpMethod.GET})
    public OkPojo facesReindexed(@RequestParam("uid") String rawUid)
    {
        DjfsUid uid = DjfsUid.cons(rawUid);
        bazingaTaskManager.schedule(new ReinitializeUserFacesTask(uid));
        return new OkPojo();
    }

    @Path(value = "/v1/albums/faces", methods = {HttpMethod.DELETE})
    public OkPojo deleteUserFaces(@RequestParam("uid") String rawUid)
    {
        DjfsUid uid = DjfsUid.cons(rawUid);
        facesAlbumManager.deleteAlbums(uid);
        return new OkPojo();
    }

    @Path(value = "/v1/albums/faces/indexer/updated", methods = {HttpMethod.GET})
    public OkPojo facesUpdated(@RequestParam("uid") String rawUid, @RequestParam("faceVersion") String version)
    {
        DjfsUid uid = DjfsUid.cons(rawUid);
        bazingaTaskManager.schedule(new UpdateUserFacesTask(uid));
        return new OkPojo();
    }

    @SynchronizedToAlbumLocked
    @WithThreadLocalCache
    @CheckBlocked
    @Path(value = "/v1/albums/album_remove", methods = {HttpMethod.DELETE})
    @TranslateExceptionToLegacy
    public void albumRemove(
            @RequestParam("uid") String rawUid,
            @RequestParam("albumId") String rawAlbumId
    )
    {
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid);
        ObjectId albumId = new ObjectId(rawAlbumId);
        Album album = albumDao.findAlbum(uid, albumId)
                .getOrThrow(() -> new AlbumsNotFoundException("Can't remove not existing album"));

        switch (album.getType()) {
            case PERSONAL: {
                personalAlbumManager.removeAlbum(uid, album,
                        AlbumRemoveCallbacks.defaultWithLogging(eventHistoryLogger));
                break;
            }
            case FACES: {
                facesAlbumManager.removeAlbum(uid, album,
                        AlbumRemoveCallbacks.defaultWithLogging(eventHistoryLogger));
                break;
            }
            case FAVORITES: {
                throw new AlbumUnableToDeleteException("Unable to delete favorites album");
            }
            default:
                throw new DjfsNotImplementedException();
        }
    }

    public ListF<AlbumItemPojo> getAlbumItemsWithContext(ListF<Tuple2<AlbumItem, FileDjfsResource>> items, Album album,
                                                         FormattingContext context)
    {
        ListF<AlbumItemPojo> result = Cf.arrayList();
        for (Tuple2<AlbumItem, FileDjfsResource> item: items) {
            ResourcePojo resourcePojo = resourcePojoBuilder.build(item._2, context);
            AlbumItemPojo itemPojo = AlbumItemPojo.fromResource(item._1, resourcePojo, album);
            result.add(itemPojo);
        }
        return result;
    }

    public FormattingContext getContext(DjfsUid uid, Option<String> metaField,
                                        Option<PreviewFormattingOptions> previewOptionsO, ReadResourceEndpoint endpoint)
    {
        UserData user = userDao.findExistingAndNotBlocked(uid);
        Option<SetF<String>> metaFields = metaField.map(this::parseMetaField);
        PreviewFormattingOptions previewOptions = previewOptionsO.getOrElse(PreviewFormattingOptions.EMPTY);
        FormattingContext context = new FormattingContext(
                user, metaFields, previewOptions, endpoint, Option.empty(), user.getId()
        );
        return context;
    }

    public AlbumPojo getAlbumWithMeta(DjfsUid uid, Album album, FormattingContext context)
    {
        // Reread album data
        album = albumDao.findAlbum(uid, album.getId()).get();

        ListF<AlbumItem> allItems = albumItemDao.getAllAlbumItems(uid, Cf.list(album.getId()), AlbumUtils.ALBUM_SIZE_LIMIT);
        UserPojo userPojo = publicationManager.constructUserPojo(UserData.cons(uid));

        if (album.getCoverId().isPresent() && allItems.isNotEmpty()) {
            Option<AlbumItem> currentCover0 = albumItemDao.findByItemId(uid, album.getCoverId().get().toHexString());
            if (currentCover0.isPresent()) {
                AlbumItem currentCover = currentCover0.get();

                ListF<DjfsResource> resource = djfsResourceDao.find(DjfsResourceId.cons(album.getUid(), currentCover.getObjectId()));

                if (resource.isNotEmpty()) {
                    FileDjfsResource currentCoverResource = (FileDjfsResource) resource.first();
                    ResourcePojo currentCoverPojo = resourcePojoBuilder.build(currentCoverResource, context);
                    AlbumItemPojo coverItemPojo = AlbumItemPojo.fromResource(currentCover, currentCoverPojo, album);

                    return AlbumPojo.fromAlbum(album, userPojo, Option.of(coverItemPojo), allItems.isEmpty(),
                            Option.of(currentCoverResource), mpfsClient);
                }
            }
        }
        return AlbumPojo.fromAlbum(album, userPojo, Option.empty(), allItems.isEmpty(), Option.empty(), mpfsClient);
    };

    @SynchronizedToAlbumLocked
    @WithThreadLocalCache
    @CheckBlocked
    @Path(value = "/v1/albums/album_publish", methods = {HttpMethod.POST, HttpMethod.PATCH})
    @TranslateExceptionToLegacy
    public AlbumPojo albumPublish(
            @RequestParam("uid") String rawUid,
            @RequestParam("albumId") String rawAlbumId,
            @RequestParam("meta") Option<String> metaField
    )
    {
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid);
        ObjectId albumId = new ObjectId(rawAlbumId);
        Album album = albumDao.findAlbum(uid, albumId)
                .getOrThrow(() -> new AlbumsNotFoundException("Can't publish not existing album"));

        if (album.getType() == AlbumType.PERSONAL) {
            personalAlbumManager.changePublicity(uid, album, true,
                    AlbumSetAttrCallbacks.defaultWithLogging(eventHistoryLogger));
        } else {
            throw new DjfsNotImplementedException();
        }
        FormattingContext context = getContext(uid, metaField, Option.empty(), ReadResourceEndpoint.ALBUM_SET_ATTR);
        return getAlbumWithMeta(uid, album, context);
    };

    @SynchronizedToAlbumLocked
    @WithThreadLocalCache
    @CheckBlocked
    @Path(value = "/v1/albums/album_unpublish", methods = {HttpMethod.POST, HttpMethod.PATCH})
    @TranslateExceptionToLegacy
    public AlbumPojo albumUnpublish(
            @RequestParam("uid") String rawUid,
            @RequestParam("albumId") String rawAlbumId,
            @RequestParam("meta") Option<String> metaField
    )
    {
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid);
        ObjectId albumId = new ObjectId(rawAlbumId);
        Album album = albumDao.findAlbum(uid, albumId)
                .getOrThrow(() -> new AlbumsNotFoundException("Can't unpublish not existing album"));
        if (album.getType() == AlbumType.PERSONAL) {
            personalAlbumManager.changePublicity(uid, album, false,
                    AlbumSetAttrCallbacks.defaultWithLogging(eventHistoryLogger));
        } else {
            throw new DjfsNotImplementedException();
        }
        FormattingContext context = getContext(uid, metaField, Option.empty(), ReadResourceEndpoint.ALBUM_SET_ATTR);
        return getAlbumWithMeta(uid, album, context);
    };

    @SynchronizedToAlbumLocked
    @WithThreadLocalCache
    @CheckBlocked
    @Path(value = "/v1/albums/album_set_attr", methods = {HttpMethod.PATCH})
    @TranslateExceptionToLegacy
    public AlbumPojo albumSetAttr(
            @RequestParam("uid") String rawUid,
            @RequestParam("albumId") String rawAlbumId,
            @RequestParam(value="title", required=false) Option<String> title,
            @RequestParam(value="cover", required=false) Option<String> rawCover,
            @RequestParam(value="coverOffsetY", required=false) Option<Double> coverOffsetY,
            @RequestParam(value="layout", required=false) Option<String> layout,
            @RequestParam(value="flags", required=false) Option<String> rawFlags,
            @RequestParam(value="fotkiAlbumId", required=false) Option<Long> fotkiAlbumId,
            @RequestParam(value="description", required=false) Option<String> description,
            @RequestParam("meta") Option<String> metaField,
            @RequestParam(value="previewSize", required=false) Option<String> previewSize,
            @RequestParam(value="previewCrop", required=false) Option<String> previewCrop,
            @RequestParam(value="previewQuality", required=false) Option<String> previewQuality,
            @RequestParam(value="previewAllowBigSize", required=false) Option<String> previewAllowBigSize
    )
    {
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid);
        ObjectId albumId = new ObjectId(rawAlbumId);
        Album album = albumDao.findAlbum(uid, albumId)
                .getOrThrow(() -> new AlbumsNotFoundException("Can't change not existing album"));

        if (title.isPresent() && title.get().isEmpty()) {
            title = Option.empty();
        }

        switch (album.getType()) {
            case PERSONAL: {
                personalAlbumManager.setAttrs(
                        uid, album, title, rawCover, coverOffsetY, layout, description, rawFlags, fotkiAlbumId,
                        AlbumSetAttrCallbacks.defaultWithLogging(eventHistoryLogger)
                );
                break;
            }
            case FACES: {
                facesAlbumManager.setAttrs(
                        uid, album, title, rawCover, coverOffsetY, layout, description, rawFlags,
                        AlbumSetAttrCallbacks.defaultWithLogging(eventHistoryLogger)
                );
                break;
            }
            case GEO:{
                geoAlbumManager.setAttrs(
                        uid, album, rawCover, coverOffsetY, layout, description, rawFlags,
                        AlbumSetAttrCallbacks.defaultWithLogging(eventHistoryLogger)
                );
                break;
            }
            case FAVORITES: {
                favoritesAlbumManager.setAttrs(
                        uid, album, layout, description, rawFlags,
                        AlbumSetAttrCallbacks.defaultWithLogging(eventHistoryLogger)
                );
                break;
            }
            default:
                throw new DjfsNotImplementedException();
        }
        PreviewFormattingOptions previewOptions = new PreviewFormattingOptions(previewSize, previewCrop,
                previewQuality, previewAllowBigSize, Option.empty(), Option.empty());
        FormattingContext context = getContext(uid, metaField, Option.of(previewOptions), ReadResourceEndpoint.ALBUM_SET_ATTR);
        return getAlbumWithMeta(uid, album, context);
    }

    @SynchronizedToAlbumLocked
    @CheckBlocked
    @Path(value = "/v1/albums/album_append_items", methods = {HttpMethod.POST, HttpMethod.GET})
    @TranslateExceptionToLegacy
    public AlbumAppendPojo albumAppendItems(
            @BoundJsonListByBender ListF<String> rawResourcePaths,
            @RequestParam("uid") String rawUid,
            @RequestParam("albumId") String rawAlbumId,
            @RequestParam("ifNotExists") String rawIfNotExists,
            @RequestParam("meta") Option<String> metaField
    )
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        UserData user = userDao.findExistingAndNotBlocked(uid);

        ObjectId albumId = new ObjectId(rawAlbumId);
        Album album = albumDao.findAlbum(uid, albumId)
                .getOrThrow(() -> new AlbumsNotFoundException("Can't append items to not existing album"));
        boolean ifNotExists = rawIfNotExists.equals("1");
        ListF<DjfsResourcePath> paths = rawResourcePaths.filter(clientInputDataProcessor::isValidPath)
                .map(rawPath -> getPath(rawPath, uid)).filter(Option::isPresent)
                .map(Option::get);
        DjfsPrincipal principal = DjfsPrincipal.cons(user);
        ListF<DjfsResource> resources = filesystem.findByPaths(
                principal, paths, Option.of(ReadPreference.secondaryPreferred())
        );

        ListF<Tuple2<AlbumItem, FileDjfsResource>> items =
                personalAlbumManager.addPhotosToAlbum(
                        resources,
                        album,
                        ifNotExists,
                        AlbumAppendCallbacks.defaultWithLogging(eventHistoryLogger),
                        false,
                        false
                );

        FormattingContext context = getContext(uid, metaField, Option.empty(), ReadResourceEndpoint.ALBUM_SET_ATTR);
        AlbumPojo albumPojo = getAlbumWithMeta(uid, album, context);
        ListF<AlbumItemPojo> allItems = getAlbumItemsWithContext(items, album, context);
        return new AlbumAppendPojo(albumPojo, allItems);
    }

    private Option<String> resolveRawAlbumIdO(DjfsUid uid, Option<String> rawAlbumIdO, String rawItemId) {
        if (rawAlbumIdO.isPresent()) {
            return rawAlbumIdO;
        }
        return albumItemDao.findByItemId(uid, rawItemId).map(item -> item.getAlbumId().toHexString());
    }

    @SynchronizedToAlbumLocked
    @CheckBlocked
    @Path(value = "/v1/albums/album_item_remove", methods = {HttpMethod.DELETE})
    @TranslateExceptionToLegacy
    public OkPojo albumItemRemove(
            @RequestParam("uid") String rawUid,
            @RequestParam("albumId") Option<String> rawAlbumIdO,
            @RequestParam("itemId") String rawItemId
    ) {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        Option<String> albumIdO = resolveRawAlbumIdO(uid, rawAlbumIdO, rawItemId);
        if (!albumIdO.isPresent()) {
            return new OkPojo();
        }
        ObjectId albumId = new ObjectId(albumIdO.get());
        Album album = albumDao.findAlbum(uid, albumId)
                .getOrThrow(() -> new AlbumsNotFoundException("Can't remove items from not existing album"));

        switch (album.getType()) {
            case FAVORITES:
            case FACES:
            case PERSONAL: {
                // Да, для избранного всё должно происходить как для персональных
                personalAlbumManager.removePhotoFromAlbum(uid, album, rawItemId,
                        AlbumItemRemoveCallbacks.defaultWithLogging(eventHistoryLogger), false);
                break;
            }
            default:
                throw new DjfsNotImplementedException();
        }
        return new OkPojo();
    }

    @CheckBlocked
    @Path(value = "/v1/albums/faces/merge", methods = {HttpMethod.POST})
    public AsyncOperationResultPojo asyncMergeFaces(
            @RequestParam("uid") String rawUid,
            @RequestParam("srcAlbumId") String rawSrcAlbumId,
            @RequestParam("dstAlbumId") String rawDstAlbumId,
            @SpecialParam HttpServletRequestX req)
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        if (rawSrcAlbumId.equals(rawDstAlbumId)) {
            throw new A3ExceptionWithStatus(
                    "Albums ids collision", "Can not merge the album with itself", HttpStatus.SC_400_BAD_REQUEST);
        }

        ObjectId srcAlbumId = new ObjectId(rawSrcAlbumId);
        ObjectId dstAlbumId = new ObjectId(rawDstAlbumId);

        long userDiskVersion = UserData.getFromRequest(req).getVersion().getOrElse(0L);
        Operation operation = albumMergeManager.mergeFaceAlbumsAsync(uid, srcAlbumId, dstAlbumId);

        return new AsyncOperationResultPojo(operation, userDiskVersion);
    }

    @Path(value = "/v1/albums/faces/refuse-and-delete", methods = {HttpMethod.POST})
    public void refuseAndDeleteFaces(@RequestParam("uid") String rawUid) {
        final DjfsUid uid = DjfsUid.cons(rawUid);
        facesAlbumManager.receiveUserRefusal(uid);
        bazingaTaskManager.schedule(new DeleteFacesTask(uid));
    }

    private SetF<String> parseMetaField(String metaField) {
        return Cf.x(metaField.split(",")).filterNot(String::isEmpty).unique();
    }

    private Option<DjfsResourcePath> getPath(String rawPath, DjfsUid uid) {
        if (DjfsResourcePath.containsUid(rawPath)) {
            try {
                DjfsResourcePath path = DjfsResourcePath.cons(rawPath);
                if (path.getUid().equals(uid)) {  // MPFS not return shared resources to participant by owner paths
                    return Option.of(path);
                }
            } catch (InvalidDjfsResourcePathException e) {
                return Option.empty();
            }
        }
        try {
            return Option.of(DjfsResourcePath.cons(uid, rawPath));
        } catch (InvalidDjfsResourcePathException e) {
            return Option.empty();
        }
    }
}
