package ru.yandex.chemodan.app.djfs.core.album;

import lombok.Value;

import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author alexandret
 */

@Bendable
@Value
public class AlbumItemFaceCoordinates {
    final static double TOP_OFFSET_RELATIVE = 0.35;
    final static double BOTTOM_OFFSET_RELATIVE = 0.1;
    final static double HEIGHT_OFFSET_RELATIVE = TOP_OFFSET_RELATIVE + BOTTOM_OFFSET_RELATIVE;
    final static double COVER_MAX_WIDTH = 750;
    final static double COVER_MAX_HEIGHT = 600;

    @BenderPart
    AlbumItemFaceCoordinate leftTopAngle;

    @BenderPart
    AlbumItemFaceCoordinate rightBottomAngle;

    /** Relative resolution (ratio of actual pixel number to expected) of this photo if it will be an album cover */
    double coverResolutionEstimate;

    public AlbumItemFaceCoordinates(FaceInfo faceInfo) {
        // ЕСЛИ ПРАВИШЬ ТУТ ПОПРАВЬ И В MPFS
        // Логика вычисления leftTopAngle и rightBottomAngle временно (пока не закопаем альбомы в MPFS) продублирована
        // https://a.yandex-team.ru/arc/trunk/arcadia/disk/mpfs/lib/mpfs/core/albums/logic/common.py?rev=7313582#L139.
        // coverResolutionEstimate нужно только внутри DJFS для выбора обложки альбому-лицу

        double xRelative = faceInfo.getFaceCoordX();
        double yRelative = faceInfo.getFaceCoordY();
        double faceWidthRelative = faceInfo.getFaceWidth();
        double faceHeightRelative = faceInfo.getFaceHeight();
        double photoWidthPx = faceInfo.getWidth();
        double photoHeightPx = faceInfo.getHeight();

        double widthToHeight = 3.0 / 4.0;
        double heightToWidth = 4.0 / 3.0;

        double photoHeightToWidth = photoHeightPx / photoWidthPx;
        double photoWidthToHeight = photoWidthPx / photoHeightPx;

        double avatarMaxWidthRel = 1d;
        double avatarMaxHeightRel = 1d;

        boolean resizeWidthFirst = false;
        if (photoWidthPx / 4.0 < photoHeightPx / 3.0) {
            avatarMaxHeightRel = photoWidthToHeight * widthToHeight;
            resizeWidthFirst = true;
        } else {
            avatarMaxWidthRel = photoHeightToWidth * heightToWidth;
        }

        double topOffset = TOP_OFFSET_RELATIVE * faceHeightRelative;
        double bottomOffset = BOTTOM_OFFSET_RELATIVE * faceHeightRelative;
        double topRelative = yRelative - topOffset;
        double bottomRelative = yRelative + faceHeightRelative + bottomOffset;
        double avatarHeightPx = (bottomRelative - topRelative) * photoHeightPx;
        double avatarWidthPx = heightToWidth * avatarHeightPx;
        double avatarWidthRelative = avatarWidthPx / photoWidthPx;
        double widthOffset = (avatarWidthRelative - faceWidthRelative) / 2.0;

        double leftRelative = xRelative - widthOffset;
        double rightRelative = xRelative + faceWidthRelative + widthOffset;

        AlbumItemFaceBorders verticalBorders = new AlbumItemFaceBorders(leftRelative, rightRelative);
        AlbumItemFaceBorders horizontalBorders = new AlbumItemFaceBorders(topRelative, bottomRelative);

        if (resizeWidthFirst) {
            verticalBorders = verticalBorders.shiftBorders();
            if (verticalBorders.isNeedResize()) {
                double heightOffset = avatarMaxHeightRel - faceHeightRelative;

                if (heightOffset >= 0) {
                    bottomOffset = heightOffset * BOTTOM_OFFSET_RELATIVE / HEIGHT_OFFSET_RELATIVE;
                    topOffset = heightOffset - bottomOffset;
                } else {
                    topOffset = heightOffset * BOTTOM_OFFSET_RELATIVE / HEIGHT_OFFSET_RELATIVE;
                    bottomOffset = heightOffset - topOffset;
                }
                topRelative = yRelative - topOffset;
                bottomRelative = yRelative + faceHeightRelative + bottomOffset;
                horizontalBorders = new AlbumItemFaceBorders(topRelative, bottomRelative);
            }
            horizontalBorders = horizontalBorders.shiftBorders();
        } else {
            horizontalBorders = horizontalBorders.shiftBorders();
            if (horizontalBorders.isNeedResize()) {
                widthOffset = (avatarMaxWidthRel - faceWidthRelative) / 2.0;
                leftRelative = xRelative - widthOffset;
                rightRelative = xRelative + faceWidthRelative + widthOffset;
                verticalBorders = new AlbumItemFaceBorders(leftRelative, rightRelative);
            }
            verticalBorders = verticalBorders.shiftBorders();
        }

        this.leftTopAngle = new AlbumItemFaceCoordinate(verticalBorders.getBorderMin(), horizontalBorders.getBorderMin());
        this.rightBottomAngle = new AlbumItemFaceCoordinate(verticalBorders.getBorderMax(), horizontalBorders.getBorderMax());

        final double widthPx = verticalBorders.getPxSize(photoWidthPx);
        final double heightPx = horizontalBorders.getPxSize(photoHeightPx);
        // worst relative resolutions
        // Math.min because size can be smaller but cannot be greater
        final double horizontalCoverResolutionEstimate = Math.min(1, widthPx / COVER_MAX_WIDTH);
        final double verticalCoverResolutionEstimate = Math.min(1, heightPx / COVER_MAX_HEIGHT);
        // worst of horizontal and vertical
        this.coverResolutionEstimate = Math.min(horizontalCoverResolutionEstimate, verticalCoverResolutionEstimate);
    }
}
