package ru.yandex.chemodan.app.djfs.core.album;

import org.joda.time.ReadableInstant;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;

import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.MarshallerContext;
import ru.yandex.misc.bender.serialize.ToFieldMarshallerSupport;

import static org.joda.time.format.ISODateTimeFormat.date;


public class AlbumItemFieldValueJsonSerializers {
    public static final String TYPE = "type";

    private static final DateTimeFormatter dateTimeFormatter = directDateTimeFormatter();

    public static ToFieldMarshallerSupport consMarshaller() {
        return new ToFieldMarshallerSupport() {
            @Override
            public void writeJsonToField(BenderJsonWriter writer, Object fieldValue, MarshallerContext context) {
                writeJson(writer, fieldValue);
            }

            protected void writeJson(BenderJsonWriter json, Object o) {
                AlbumItemFieldValueJsonSerializers.writeJson(json, (AlbumItemFieldValue) o);
            }
        };
    }

    public static void writeJson(BenderJsonWriter json, AlbumItemFieldValue field) {
        json.writeObjectStart();
        json.writeFieldName(TYPE);

        String typeName = field.fieldType.name().toLowerCase();

        json.writeString(typeName);
        json.writeFieldName(typeName);

        switch (field.fieldType) {
            case STRING:
                writeString(json, field);
                break;
            case INTEGER:
                writeInteger(json, field);
                break;
            case BOOLEAN:
                writeBoolean(json, field);
                break;
            case DOUBLE:
                writeDecimal(json, field);
                break;
            case DATETIME:
                writeDatetime(json, field);
                break;
            default:
                throw new NotImplementedException("Field type " + field.fieldType.toString() + " is not implemented");
        }

        json.writeObjectEnd();
    }

    public static void writeBoolean(BenderJsonWriter json, AlbumItemFieldValue field) {
        json.writeBoolean((Boolean) field.value);
    }

    public static void writeInteger(BenderJsonWriter json, AlbumItemFieldValue field) {
        json.writeNumber(((Number) field.value).longValue());
    }

    public static void writeDecimal(BenderJsonWriter json, AlbumItemFieldValue field) {
        json.writeNumber(((Number) field.value).doubleValue());
    }

    public static void writeString(BenderJsonWriter json, AlbumItemFieldValue field) {
        json.writeString(field.value.toString());
    }

    public static void writeDatetime(BenderJsonWriter json, AlbumItemFieldValue field) {
        json.writeString(dateTimeFormatter.print((ReadableInstant) field.value));
    }

    private AlbumItemFieldValueJsonSerializers() {}

    private static DateTimeFormatter directDateTimeFormatter() {
        return new DateTimeFormatterBuilder()
                .append(date())
                .appendLiteral('T')
                .appendHourOfDay(2)
                .appendLiteral(':')
                .appendMinuteOfHour(2)
                .appendLiteral(':')
                .appendSecondOfMinute(2)
                .appendLiteral('.')
                .appendFractionOfSecond(6, 6)
                .appendTimeZoneOffset("+00:00", true, 2, 2)
                .toFormatter();
    }
}
