package ru.yandex.chemodan.app.djfs.core.album;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.notification.XivaData;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGeneratorAlbumUtils;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.lang.Validate;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class AlbumUpdateManager {
    protected final AlbumDao albumDao;
    protected final AlbumDeltaDao albumDeltaDao;
    protected final AlbumItemDao albumItemDao;
    protected final TransactionUtils transactionUtils;
    protected final XivaPushGenerator xivaPushGenerator;

    protected long getAndLockCurrentRevision(DjfsUid uid) {
        Option<Long> currentRevisionO = albumDeltaDao.getCurrentRevisionWithLock(uid);

        if (currentRevisionO.isPresent()) {
            return currentRevisionO.get();
        }

        if (albumDeltaDao.tryInitializeCurrentRevision(uid)) {
            return 0;
        }

        return albumDeltaDao.getCurrentRevisionWithLock(uid).get();
    }

    protected class DeleteAlbumOperation extends ConsistentOperation {
        private final boolean sendPushes;

        public DeleteAlbumOperation(Album album) {
            this(album, true);
        }

        public DeleteAlbumOperation(Album album, boolean sendPushes) {
            super(album.getUid(), album.getType(), Cf.set(album), false);
            this.sendPushes = sendPushes;
        }

        @Override
        protected OperationResult updateAndGetChanges() {
            Album album = albums.single();
            return albumDao.delete(album)
                    ? new OperationResult(
                            Cf.list(AlbumUtils.createDeleteAlbumChange(album)),
                            Option.when(sendPushes, () -> XivaPushGeneratorAlbumUtils.albumRemoved(uid, album)))
                    : OperationResult.empty();
        }
    }

    @AllArgsConstructor
    protected class ConsistentOperation {
        protected final DjfsUid uid;
        protected final AlbumType albumType;
        protected final SetF<Album> albums;
        protected final boolean updateDateModified;
        private final ListF<Function0<OperationResult>> updateAndGetChangesFs;

        public ConsistentOperation(DjfsUid uid, AlbumType albumType,
                                   SetF<Album> albums, boolean updateDateModified)
        {
            Validate.forAll(albums, album -> album.getType() == albumType);
            this.uid = uid;
            this.albumType = albumType;
            this.albums = albums;
            this.updateDateModified = updateDateModified;
            this.updateAndGetChangesFs = Cf.list(this::updateAndGetChanges);
        }

        protected OperationResult updateAndGetChanges() {
            return OperationResult.empty();
        }

        private OperationResult getChanges() {
            return updateAndGetChangesFs.map(Function0::apply).reduceLeft(OperationResult::plus);
        }

        public long execute() {
            OperationTransactionResult result = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
                OperationResult changesAndPushes = getChanges();
                ListF<AlbumDeltaChange> changes = changesAndPushes.changes;

                if (changes.isEmpty()) {
                    return new OperationTransactionResult(getAndLockCurrentRevision(uid), Cf.list());
                }
                long revision = getAndLockCurrentRevision(uid) + 1;

                Instant now = Instant.now();

                SetF<ObjectId> removedAlbums =
                        changes.filter(AlbumUtils::isAlbumDeleteChange).map(a -> new ObjectId(a.getRecordId())).unique();
                SetF<Album> remainingAlbums = albums.filterNot(album -> removedAlbums.containsTs(album.getId()));

                ListF<AlbumDeltaChange> remainingChanges = changes
                        .plus(remainingAlbums.map(album -> AlbumUtils.createSetDateModifyChange(album, now)))
                        .filterNot(
                                change -> AlbumUtils.isAlbumChange(change) &&
                                        !AlbumUtils.isDeleteChange(change) &&
                                        removedAlbums.containsTs(new ObjectId(change.recordId)));

                AlbumDelta delta = AlbumUtils.createAlbumDelta(uid, albumType, remainingChanges);

                albumDao.updateAlbumsRevision(
                        remainingAlbums.toList(), uid, revision, Option.when(updateDateModified, now));
                albumDeltaDao.insert(delta.withRevision(revision));
                albumDeltaDao.updateCurrentRevision(uid, revision);
                return new OperationTransactionResult(revision, changesAndPushes.pushes);
            });
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, result.revision);
            result.pushes.forEach(xivaPushGenerator::sendPush);
            return result.revision;
        }

        public ConsistentOperation andThen(ConsistentOperation other) {
            Validate.equals(uid, other.uid);
            Validate.equals(albumType, other.albumType);
            return new ConsistentOperation(uid, albumType,
                    albums.plus(other.albums),
                    updateDateModified || other.updateDateModified,
                    updateAndGetChangesFs.plus(other.updateAndGetChangesFs));
        }
    }

    @AllArgsConstructor
    @Data
    private static class OperationTransactionResult {
        private final long revision;
        private final ListF<XivaData> pushes;
    }

    @AllArgsConstructor
    @Data
    protected static class OperationResult {
        private final ListF<AlbumDeltaChange> changes;
        private final ListF<XivaData> pushes;

        public static OperationResult empty() {
            return new OperationResult(Cf.list(), Cf.list());
        }

        public OperationResult plus(OperationResult other) {
            return new OperationResult(this.changes.plus(other.changes), this.pushes.plus(other.pushes));
        }
    }
}
