package ru.yandex.chemodan.app.djfs.core.album;

import lombok.RequiredArgsConstructor;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;

@RequiredArgsConstructor
public class AlbumUtils {
    static final int GEO_ALBUM_SIZE_LIMIT = 20000;
    static final int ALBUM_SIZE_LIMIT = 20000;

    public static ListF<AlbumItemField> generateAllFields(AlbumItemFaceCoordinates coordinates) {
        return Cf.list(
                AlbumItemField.doubleField("face_left", coordinates.getLeftTopAngle().getX()),
                AlbumItemField.doubleField("face_top", coordinates.getLeftTopAngle().getY()),
                AlbumItemField.doubleField("face_right", coordinates.getRightBottomAngle().getX()),
                AlbumItemField.doubleField("face_bottom", coordinates.getRightBottomAngle().getY())
        );
    }

    public static ListF<AlbumItemField> generateAllFields(AlbumItem item, DjfsUid resourceOwner) {
        return Cf.list(
                AlbumItemField.stringField("album_id", item.getAlbumId().toHexString()),
                AlbumItemField.stringField("description", item.getDescription().getOrElse("")),
                AlbumItemField.doubleField("order_index", item.getOrderIndex().getOrElse(0d)),
                AlbumItemField.stringField("group_id", item.getGroupId().getOrElse("")),
                AlbumItemField.stringField("obj_id", item.getObjectId()),
                AlbumItemField.stringField("obj_type", item.getObjectType().value()),
                AlbumItemField.stringField("obj_resource_id",
                        DjfsResourceId.cons(resourceOwner, item.getObjectId()).getValue())
        ).plus(item.getFaceInfo().map(FaceInfo::getCoordinates).flatMap(AlbumUtils::generateAllFields));
    }

    public static ListF<AlbumDeltaFieldChange> generateSetChangeFields(AlbumItem item) {
        ListF<AlbumItemField> itemFields = generateAllFields(item, item.getUid());
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

    public static ListF<AlbumItemField> generateAllFields(Album album, int itemsCount,
                                                          Option<DjfsResourceId> coverResourceId)
    {
        return Cf.list(
                AlbumItemField.stringField("title", album.getTitle()),
                AlbumItemField.stringField("description", album.getDescription().getOrElse("")),
                AlbumItemField.stringField("type", album.getType().value()),
                AlbumItemField.stringField("cover_id", album.getCoverId().map(ObjectId::toHexString).getOrElse("")),
                AlbumItemField.stringField("cover_resource_id", coverResourceId.map(DjfsResourceId::getValue).getOrElse("")),
                AlbumItemField.doubleField("cover_offset", album.getCoverOffsetY().getOrElse(0d)),
                AlbumItemField.stringField("public_key", album.getPublicKey().getOrElse("")),
                AlbumItemField.stringField("public_url", album.getPublicUrl().getOrElse("")),
                AlbumItemField.stringField("short_url", album.getShortUrl().getOrElse("")),
                AlbumItemField.booleanField("is_public", album.isPublic()),
                AlbumItemField.booleanField("is_blocked", album.isBlocked()),
                AlbumItemField.stringField("block_reason", album.getBlockReason().getOrElse("")),
                AlbumItemField.stringField("layout", album.getLayout().map(AlbumLayoutType::value).getOrElse("")),
                AlbumItemField.longField("geo_id", album.getGeoId().getOrElse(0L)),
                AlbumItemField.integerField("items_count", itemsCount),
                AlbumItemField.booleanField("is_visible", !album.isHidden()),
                AlbumItemField.instantField("modified", album.getDateModified().getOrElse(new Instant(0)))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetChangeFields(
            Album album, int itemsCount, Option<DjfsResourceId> coverResourceId)
    {
        ListF<AlbumItemField> itemFields = generateAllFields(album, itemsCount, coverResourceId);
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetVisibilityChangeFields(boolean isVisible)
    {
        AlbumItemField visibilityField = AlbumItemField.booleanField("is_visible", isVisible);
        return Cf.list(
                new AlbumDeltaFieldChange(visibilityField.fieldId, AlbumDeltaFieldChangeType.SET,
                        Option.of(visibilityField.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetItemsCountChangeFields(int newItemsCount)
    {
        AlbumItemField itemsCountField = AlbumItemField.integerField("items_count", newItemsCount);
        return Cf.list(
                new AlbumDeltaFieldChange(itemsCountField.fieldId, AlbumDeltaFieldChangeType.SET,
                        Option.of(itemsCountField.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetCoverChangeFields(ObjectId coverId,
                                                                            DjfsResourceId coverResourceId)
    {
        ListF<AlbumItemField> itemFields = Cf.list(
                AlbumItemField.stringField("cover_id", coverId.toHexString()),
                AlbumItemField.stringField("cover_resource_id", coverResourceId.getValue())
        );
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetTitleChangeFields(String title)
    {
        ListF<AlbumItemField> itemFields = Cf.list(
                AlbumItemField.stringField("title", title)
        );
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetLayoutChangeFields(AlbumLayoutType layout)
    {
        ListF<AlbumItemField> itemFields = Cf.list(
                AlbumItemField.stringField("layout", layout.value())
        );
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetDescriptionChangeFields(String description)
    {
        ListF<AlbumItemField> itemFields = Cf.list(
                AlbumItemField.stringField("description", description)
        );
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

//    public static ListF<AlbumDeltaFieldChange> generateSetFotkiAlbumIdChangeFields(String fotkiAlbumId)
//    {
//        По идее тут bigint
//        ListF<AlbumItemField> itemFields = Cf.list(
//                AlbumItemField.stringField("fotki_album_id", fotkiAlbumId)
//        );
//        return itemFields.map(
//                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
//        );
//    }

    public static ListF<AlbumDeltaFieldChange> generateSetCoverOffsetYChangeFields(double coverOffsetY)
    {
        ListF<AlbumItemField> itemFields = Cf.list(
                AlbumItemField.doubleField("cover_offset_y", coverOffsetY)
        );
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }

    public static ListF<AlbumDeltaFieldChange> generateSetPublicChangeFields(boolean isPublic)
    {
        ListF<AlbumItemField> itemFields = Cf.list(
                AlbumItemField.booleanField("is_public", isPublic)
        );
        return itemFields.map(
                x -> new AlbumDeltaFieldChange(x.fieldId, AlbumDeltaFieldChangeType.SET, Option.of(x.value))
        );
    }


    public static ListF<AlbumDeltaFieldChange> generateSetDateModifyChangeFields(Instant dateModify) {
        AlbumItemField dateModifyField = AlbumItemField.instantField("modified", dateModify);
        return Cf.list(
                new AlbumDeltaFieldChange(dateModifyField.fieldId, AlbumDeltaFieldChangeType.SET,
                        Option.of(dateModifyField.value))
        );
    }

    private static AlbumDelta createAlbumChangeDelta(AlbumDeltaChangeType type, Album album, AlbumItem item,
                                                     ListF<AlbumDeltaFieldChange> changes, long revision)
    {
        return AlbumDelta.builder()
                .uid(album.getUid())
                .id(UuidUtils.randomToHexString())
                .revision(Option.of(revision))
                .changes(Cf.list(createAlbumItemChange(type, item, changes)))
                .dateCreated(Option.of(Instant.now()))
                .albumType(Option.of(album.getType()))
                .build();
    }

    public static AlbumDelta createInsertAlbumItemDelta(Album album, AlbumItem item, long revision) {
        return createAlbumChangeDelta(
                AlbumDeltaChangeType.INSERT,
                album,
                item,
                AlbumUtils.generateSetChangeFields(item),
                revision
        );
    }

    public static AlbumDelta createDeleteAlbumItemDelta(Album album, AlbumItem item, long revision) {
        return createAlbumChangeDelta(
                AlbumDeltaChangeType.DELETE,
                album,
                item,
                Cf.list(),
                revision
        );
    }

    private static AlbumDelta createIndexDelta(AlbumDeltaChangeType type, Album album,
                                               ListF<AlbumDeltaFieldChange> changes, long revision)
    {
        return createAlbumDelta(
                album.getUid(), album.getType(), Cf.list(createAlbumChange(type, album, changes)), Option.of(revision));
    }

    public static AlbumDelta createAlbumDelta(DjfsUid uid, AlbumType albumType, ListF<AlbumDeltaChange> changes) {
        return createAlbumDelta(uid, albumType, changes, Option.empty());
    }

    public static AlbumDelta createAlbumDelta(DjfsUid uid, AlbumType albumType,
                                              ListF<AlbumDeltaChange> changes, Option<Long> revision) {
        return AlbumDelta.builder()
                .uid(uid)
                .id(UuidUtils.randomToHexString())
                .revision(revision)
                .changes(changes)
                .dateCreated(Option.of(Instant.now()))
                .albumType(Option.of(albumType))
                .build();
    }

    public static AlbumDelta createInsertAlbumDelta(
            Album album, int itemsCount, Option<DjfsResourceId> coverResourceId, long revision)
    {
        return createIndexDelta(
                AlbumDeltaChangeType.INSERT,
                album,
                AlbumUtils.generateSetChangeFields(album, itemsCount, coverResourceId),
                revision
        );
    }

    public static AlbumDelta createUpdateAlbumDelta(Album album, ListF<AlbumDeltaFieldChange> changes, long revision) {
        return createIndexDelta(AlbumDeltaChangeType.UPDATE, album, changes, revision);
    }

    public static AlbumDelta createRemoveAlbumDelta(Album album, long revision) {
        return createIndexDelta(AlbumDeltaChangeType.DELETE, album, Cf.list(), revision);
    }

    public static ListF<AlbumDeltaFieldChange> selectNewCover(Album album, AlbumItemDao albumItemDao, AlbumDao albumDao) {
        ListF<AlbumItem> allItems = albumItemDao.getAllAlbumItems(
                album.getUid(), Cf.list(album.getId()), 1);

        if (allItems.isEmpty()) {
            return Cf.list();  // TODO: generate cover remove delta and sunset cover id for album???
        }

        albumDao.setCover(album, allItems.first().getId());
        return AlbumUtils.generateSetCoverChangeFields(allItems.first().getId(),
                DjfsResourceId.cons(album.getUid(), allItems.first().getObjectId()));
    }

    public static AlbumDeltaChange createAlbumItemChange(AlbumDeltaChangeType type, AlbumItem item,
                                                  ListF<AlbumDeltaFieldChange> changes)
    {
        return new AlbumDeltaChange(item.getId().toHexString(), item.getAlbumId().toHexString(), type, changes);
    }

    public static AlbumDeltaChange createAlbumChange(AlbumDeltaChangeType type, Album album,
                                                  ListF<AlbumDeltaFieldChange> changes)
    {
        return new AlbumDeltaChange(album.getId().toHexString(), CollectionId.ALBUM_LIST_COLLECTION_ID, type, changes);
    }

    public static AlbumDeltaChange createDeleteAlbumChange(Album album) {
        return createAlbumChange(AlbumDeltaChangeType.DELETE, album, Cf.list());
    }

    public static AlbumDeltaChange createSetItemsCountChange(Album album, int itemsCount) {
        return createAlbumChange(AlbumDeltaChangeType.UPDATE, album, generateSetItemsCountChangeFields(itemsCount));
    }

    public static AlbumDeltaChange createSetDateModifyChange(Album album, Instant instant) {
        return createAlbumChange(AlbumDeltaChangeType.UPDATE, album, generateSetDateModifyChangeFields(instant));
    }

    public static AlbumDeltaChange createSetTitleChange(Album album, String title) {
        return createAlbumChange(AlbumDeltaChangeType.UPDATE, album, generateSetTitleChangeFields(title));
    }

    public static boolean isAlbumDeleteChange(AlbumDeltaChange change) {
        return isDeleteChange(change) && isAlbumChange(change);
    }

    public static boolean isDeleteChange(AlbumDeltaChange change) {
        return change.getChangeType() == AlbumDeltaChangeType.DELETE;
    }

    public static boolean isAlbumChange(AlbumDeltaChange change) {
        return change.getCollectionId().equals(CollectionId.ALBUM_LIST_COLLECTION_ID);
    }

    public static AlbumDeltaChange createInsertChanges(AlbumItem item) {
        ListF<AlbumDeltaFieldChange> changes = generateSetChangeFields(item);
        return createAlbumItemChange(AlbumDeltaChangeType.INSERT, item, changes);
    }

    public static AlbumDeltaChange createDeleteChanges(AlbumItem item) {
        return createAlbumItemChange(AlbumDeltaChangeType.DELETE, item, Cf.list());
    }
}
