package ru.yandex.chemodan.app.djfs.core.album;

import lombok.Value;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchHttpClient;
import ru.yandex.chemodan.app.djfs.core.client.FaceClusterId;
import ru.yandex.chemodan.app.djfs.core.client.FaceClusterPhotos;
import ru.yandex.chemodan.app.djfs.core.client.FaceClustersSnapshot;
import ru.yandex.chemodan.app.djfs.core.client.FacesDelta;
import ru.yandex.chemodan.app.djfs.core.client.FacesDeltas;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumRemoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumSetAttrActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumappend.AlbumAppendCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumitemremove.AlbumItemRemoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumremove.AlbumRemoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumsetattr.AlbumSetAttrCallbacks;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.BlackboxUtils;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGeneratorAlbumUtils;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.FacesIndexingState;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/** Tools for working with 'faces' albums. */
public class FacesAlbumManager extends AbstractAlbumManager {
    static final SetF<String> ALLOWED_COUNTRIES = Cf.set("BY", "KZ", "RU", "UA");

    private static final Logger logger = LoggerFactory.getLogger(FacesAlbumManager.class);

    private final AlbumFaceClustersDao albumFaceClustersDao;
    private final Blackbox2 blackbox;
    private final DiskSearchHttpClient diskSearchHttpClient;
    private final UserDao userDao;

    protected FacesAlbumManager(
            AlbumDao albumDao,
            AlbumItemDao albumItemDao,
            AlbumFaceClustersDao albumFaceClustersDao,
            Blackbox2 blackbox,
            DiskSearchHttpClient diskSearchHttpClient,
            DjfsResourceDao djfsResourceDao,
            UserDao userDao,
            AlbumDeltaDao albumDeltaDao,
            XivaPushGenerator xivaPushGenerator,
            TransactionUtils transactionUtils,
            ShareInfoManager shareInfoManager
    ) {
        super(
                albumDao,
                albumItemDao,
                albumDeltaDao,
                transactionUtils,
                xivaPushGenerator,
                shareInfoManager,
                djfsResourceDao
        );
        this.albumFaceClustersDao = albumFaceClustersDao;
        this.blackbox = blackbox;
        this.diskSearchHttpClient = diskSearchHttpClient;
        this.userDao = userDao;
    }

    @Override
    public AlbumType getAlbumType() {
        return AlbumType.FACES;
    }

    /**
     Initialize the user account with 'faces' albums made of clusters from the Search service.

     WARNING: Not idempotent. Used consequently, this will not update albums.
     To reset albums, remove them and initialize again.
     */
    public void initializeAlbumsFromClusters(DjfsUid uid) {
        FaceClustersSnapshot clusters = diskSearchHttpClient.getFaceClusters(uid);

        logger.info("initializeAlbumsFromClusters().getFaceClusters for uid {}: clusters={}",
                uid.asString(), clusters.toString());

        Option<Long> clustersVersionO = userDao.getFaceClustersVersion(uid);
        if (clustersVersionO.isPresent()) {
            throw
                new EntityAlreadyExistsException(
                    "Face clusters already initialized, uid = " + uid
                    + ", known version = " + clustersVersionO.get()
                    + ", new version = " + clusters.getVersion(),
                    null
                );
        }

        final FacesIndexingState s = userDao.getFacesIndexingState(uid)._1;
        switch (s) {
            case USER_REFUSED:
                throw new UserRefusedException(uid);
            case COUNTRY_RESTRICTION:
                throw new FeatureUnavailableInUserCountryException(uid, Option.empty());
        }

        userDao.setFacesIndexingState(uid, FacesIndexingState.REINDEXED);

        int itemCount = 0;
        if (clusters.getItems().isNotEmpty()) {
            for (FaceClusterId cluster : clusters.getItems()) {
                FaceClusterPhotos photos = diskSearchHttpClient.getClusterPhotos(uid, cluster.getId());

                logger.info("initializeAlbumsFromClusters().getClusterPhotos for uid {}: cluster={}, photos={}",
                        uid.asString(), cluster.getId(), photos.toString());

                if (photos.getItems().isNotEmpty()) {
                    ListF<DjfsResourceId> duplicatingResourceIds = photos.getItems()
                            .map(FaceInfo::getResourceId).countBy().filterValues(count -> count > 1).keys();

                    ListF<FaceInfo> uniquePhotos;
                    if (duplicatingResourceIds.isNotEmpty()) {
                        uniquePhotos = photos.getItems().stableUniqueBy(FaceInfo::getResourceId);
                        logger.error("Items with duplicating resource ids in cluster {} for uid {} will be skipped: {}",
                                photos.getClusterId(), uid, duplicatingResourceIds);
                    } else {
                        uniquePhotos = photos.getItems();
                    }

                    final Tuple2<Album, Integer> albumInitializationResult =
                            initializeAlbum(uid, photos.getClusterId(), uniquePhotos, false);
                    if (albumInitializationResult._2 > 0) {
                        transactionUtils.executeInNewOrCurrentTransaction(
                                uid,
                                () -> {
                                    final long revision = getAndLockCurrentRevision(uid);
                                    albumDao.updateAlbumRevision(albumInitializationResult._1, revision + 1);
                                }
                        );
                    }
                    itemCount += albumInitializationResult._2;
                } else {
                    logger.error("Empty cluster {} for uid {}", photos.getClusterId(), uid);
                }
            }
            if (itemCount > 0) {
                transactionUtils.executeInNewOrCurrentTransaction(
                        uid,
                        () -> {
                            final long revision = getAndLockCurrentRevision(uid);
                            albumDeltaDao.updateCurrentRevision(uid, revision + 1);
                        }
                );
            } else {
                logger.error("All {} clusters are empty for uid {}", clusters.getItems().length(), uid);
            }
        } else { // no clusters
            logger.warn("No clusters for uid = {}", uid);
        }

        userDao.updateFaceClustersVersion(uid, clusters.getVersion());
        albumDeltaDao.getCurrentRevisionWithoutLock(uid)
                .ifPresent(revision -> xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, revision));
    }

    /**
     @return album created and its item count
     */
    private
    Tuple2<Album, Integer> initializeAlbum(
            DjfsUid uid, String clusterId, ListF<FaceInfo> photos, boolean notifyClients
    ) {
        final Album album = prepareAlbum(uid);

        @Value
        class Result {
            long                revision;
            int                 itemCount;
            Option<AlbumItem>   coverItemO;
        }

        Result result = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            albumDao.insert(album);
            albumFaceClustersDao.insert(uid, album.getId(), clusterId);

            Option<AlbumItem> coverItemO = Option.empty();
            int itemCount = 0;
            for (FaceInfo clusterItem : photos) {
                final Option<AlbumItem> itemO = initializeAlbumItem(album, clusterItem);
                if (!itemO.isPresent()) // item is valid
                    continue;
                ++itemCount;
                final AlbumItem item = itemO.get();
                coverItemO = coverItemO.map(coverItem -> betterCover(coverItem, item)).orElse(Option.of(item));
            }
            if (coverItemO.isPresent())
                albumDao.setCover(album, coverItemO.get().getId());
            return new Result(currentRevision, itemCount, coverItemO);
        });

        if (notifyClients) {
            final long revision = result.revision + 1;
            albumDao.updateAlbumRevision(album, revision);
            albumDeltaDao.insert(
                    AlbumUtils.createInsertAlbumDelta(
                            album,
                            result.itemCount,
                            result.coverItemO.map(x -> DjfsResourceId.cons(uid, x.getObjectId())),
                            revision
                    )
            );
            albumDeltaDao.updateCurrentRevision(uid, revision);
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, revision);
        }

        return Tuple2.tuple(album, result.itemCount);
    }

    @Override
    protected void changeCoverIfNeeded(Album album, AlbumItem item) {
        final String coverId = album.getCoverId().getOrThrow("Album must have cover").toString();
        final AlbumItem cover =
                albumItemDao.findByItemId(album.getUid(), coverId).getOrThrow("Album cover item must exist");
        final AlbumItem newCover = betterCover(cover, item);
        if (!newCover.getObjectId().equals(coverId)) {
            albumDao.setCover(album, newCover.getId());
        }
    }

    private AlbumItem betterCover(AlbumItem curCover, AlbumItem newItem) {
        final Option<FaceInfo> curFace = curCover.getFaceInfo(), newFace = newItem.getFaceInfo();
        final double curResolution = curFace
                .map(FaceInfo::getCoordinates)
                .map(AlbumItemFaceCoordinates::getCoverResolutionEstimate)
                .getOrElse(0.0);
        final double newResolution = newFace
                .map(FaceInfo::getCoordinates)
                .map(AlbumItemFaceCoordinates::getCoverResolutionEstimate)
                .getOrElse(0.0);
        final Option<Double> curAesthetic = curFace.flatMapO(FaceInfo::getAesthetic);
        final Option<Double> newAesthetic = newFace.flatMapO(FaceInfo::getAesthetic);
        return (newResolution > curResolution)
                || (newResolution == curResolution && isBetterThan(newAesthetic, curAesthetic))
                ? newItem
                : curCover;
    }

    public boolean updateAlbumsFromClusters(DjfsUid uid) {
        final Tuple2<FacesIndexingState, Instant> state = userDao.getFacesIndexingState(uid);
        if (state._1 != FacesIndexingState.REINDEXED) {
            logger.warn("Received 'updated' signal for a non-indexed account. uid = {}, state = {}", uid, state);
            return false;
        }
        boolean hasMore;
        do {
            final long knownVersion = userDao.getFaceClustersVersion(uid).getOrElse(0L);
            final FacesDeltas deltas = diskSearchHttpClient.getFacesDeltas(uid, knownVersion);

            logger.info("updateAlbumsFromClusters().deltas for uid {}: knownVersion={}, deltasVersion={}, deltas={}",
                    uid.asString(), knownVersion, deltas.getVersion(), deltas.toString());

            if (deltas.getItems().isEmpty()) {
                // just an extra 'updated' signal for already updated clusters
                return false;
            }
            if (deltas.getVersion() <= knownVersion) {
                throw new DiskSearchExpectationFailed(
                        "version", "> " + knownVersion, "" + deltas.getVersion()
                );
            }
            for (final FacesDelta delta : deltas.getItems()) {
                switch (delta.getType()) {
                    case CLUSTER_CREATED:
                    case CLUSTER_DELETED:
                        /* ignore */
                        break;
                    case ITEM_ADDED:
                        handleClusterItemAdded(uid, delta.getClusterId(), delta.getAddedResource());
                        break;
                    case ITEM_DELETED:
                        handleClusterItemDeleted(uid, delta.getClusterId(), delta.getDeletedResourceId());
                        break;
                }
            }
            userDao.updateFaceClustersVersion(uid, deltas.getVersion());
            hasMore = deltas.getHasMore();
        } while (hasMore);
        return true;
    }

    private void handleClusterItemAdded(DjfsUid uid, String clusterId, FaceInfo faceInfo) {
        final DjfsResourceId resourceId = faceInfo.getResourceId();
        if (!uid.equals(resourceId.getUid())) {
            logger.warn(
                    "File viewer is not the owner. A shared file is indexed unexpectedly. user = " + uid
                            + ", resource = " + resourceId
            );
            return;
        }

        final Option<DjfsResource> resourceO = djfsResourceDao.find(resourceId).firstO();
        if (!resourceO.isPresent()) {
            logger.warn("File not found: " + resourceId);
            return;
        }
        final DjfsResource resource = resourceO.get();

        Validate.isTrue(faceInfo.getFaceCoordX() >= 0.0 && faceInfo.getFaceCoordX() < 1.0);
        Validate.isTrue(faceInfo.getFaceCoordY() >= 0.0 && faceInfo.getFaceCoordY() < 1.0);
        Validate.isTrue(faceInfo.getFaceWidth() > 0.0 && faceInfo.getFaceWidth() <= 1.0);
        Validate.isTrue(faceInfo.getFaceHeight() > 0.0 && faceInfo.getFaceHeight() <= 1.0);
        Validate.isTrue(faceInfo.getWidth() > 0 && faceInfo.getHeight() > 0);

        final Option<ObjectId> albumIdO = albumFaceClustersDao.getAlbum(uid, clusterId);
        if (albumIdO.isPresent()) {
            final ObjectId albumId = albumIdO.get();
            final Album album = albumDao
                    .findAlbum(uid, albumId)
                    .getOrThrow(
                            "Internal error: album " + albumId + " is bound to cluster " + clusterId
                                    + " but doesn't exist"
                    );
            addPhotosToAlbum(
                    Tuple2List.fromPairs(resource, Option.of(faceInfo)),
                    album,
                    true,
                    AlbumAppendCallbacks.empty(),
                    true,
                    true
            );
        } else {
            initializeAlbum(uid, clusterId, Cf.list(faceInfo), true);
        }
    }

    private void handleClusterItemDeleted(DjfsUid uid, String clusterId, DjfsResourceId resourceId) {
        final Option<ObjectId> albumIdO = albumFaceClustersDao.getAlbum(uid, clusterId);
        if (!albumIdO.isPresent()) {
            logger.error("Deletion from non-existent album, cluster_id = " + clusterId);
            return;
        }
        final ObjectId albumId = albumIdO.get();
        final Album album = albumDao
                .findAlbum(uid, albumId)
                .getOrThrow(
                        "Internal error: album " + albumId + " is bound to cluster " + clusterId + " but doesn't exist"
                );
        final ListF<AlbumItem> items = albumItemDao.findObjectInAlbum(uid, albumId, resourceId.getFileId().getValue());
        /*
         * Для нотификации удаления передаем в пуше autoHandle = false, поскольку источником такой нотификации
         * является действие пользователя по удалению фото.
         */
        for (final AlbumItem item : items) {
            removePhotoFromAlbum(uid, album, item.getId().toString(), AlbumItemRemoveCallbacks.empty(), false);
        }
    }

    // TODO replace with Comparable.compareTo
    private boolean isBetterThan(Option<Double> a, Option<Double> b) {
        return a.isPresent() && (!b.isPresent() || a.get() > b.get());
    }

    private Option<AlbumItem> initializeAlbumItem(Album album, FaceInfo clusterItem) {
        Option<AlbumItem> albumItemO = prepareAlbumItem(album, clusterItem);
        albumItemO.forEach(albumItemDao::insert);
        return albumItemO;
    }

    private Album prepareAlbum(DjfsUid uid) {
        Instant now = Instant.now();
        return
                Album.builder()
                        .id(new ObjectId())
                        .uid(uid)
                        .type(AlbumType.FACES)
                        .title("")
                        .description(Option.empty())
                        .geoId(Option.empty())
                        .revision(Option.empty())
                        .hidden(false)
                        .dateCreated(Option.of(now))
                        .dateModified(Option.of(now))
                        .coverId(Option.empty())
                        .coverOffsetY(Option.empty())
                        .flags(Option.empty())
                        .layout(Option.empty())
                        .isBlocked(false)
                        .blockReason(Option.empty())
                        .fotkiAlbumId(Option.empty())
                        .isPublic(false)
                        .publicKey(Option.empty())
                        .publicUrl(Option.empty())
                        .shortUrl(Option.empty())
                        .socialCoverStid(Option.empty())
                        .isDescSorting(Option.empty())
                        .albumItemsSorting(Option.empty())
                        .build();
    }

    private Option<AlbumItem> prepareAlbumItem(Album album, FaceInfo faceInfo) {
        final DjfsResourceId resourceId = faceInfo.getResourceId();
        if (!album.getUid().equals(resourceId.getUid())) {
            logger.warn(
                    "File viewer is not the owner. A shared file is indexed unexpectedly. user = " + album.getUid()
                            + ", resource = " + resourceId
            );
            return Option.empty();
        }

        final Option<DjfsResource> resourceO = djfsResourceDao.find(resourceId).firstO();
        if (!resourceO.isPresent()) {
            // TODO check if resource is in /disk or /photounlim (skip /trash and other)
            logger.error("No resource found for id: " + resourceId);
            return Option.empty();
        }
        final DjfsResource resource = resourceO.get();

        if (!(resource instanceof FileDjfsResource)) {
            throw new AlbumUnableToAppendException("Resource is not a file: " + resource.getPath().getFullPath());
        }
        final FileDjfsResource file = (FileDjfsResource) resource;

        AlbumItem item = AlbumItem.cons(album, resourceId.getFileId(), itemOrderIndex(file), Option.of(faceInfo));
        return Option.of(item);
    }

    /** Delete all 'faces' albums from the user account. Used for testing, recovery reindex, or user opt-out. */
    public void deleteAlbums(DjfsUid uid) {
        albumFaceClustersDao.deleteAll(uid);
        ListF<Album> albums = albumDao.getAlbums(uid, AlbumType.FACES);
        long lastRevision = 0;
        for (Album album : albums) {
            lastRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
                final long
                        currentRevision = getAndLockCurrentRevision(uid),
                        revision = currentRevision + 1;
                albumItemDao.removeAllItemsFromAlbum(uid, album.getId());
                albumDao.delete(album);
                albumDeltaDao.insert(AlbumUtils.createRemoveAlbumDelta(album, revision));
                albumDeltaDao.updateCurrentRevision(uid, revision);
                return revision;
            });
            // no push for web to suppress pop-up notifications
        }
        xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, lastRevision);
        userDao.removeFaceClustersVersion(uid);
    }

    public void setAttrs(DjfsUid uid, Album album, Option<String> titleO, Option<String> rawCoverO,
                         Option<Double> coverOffsetYO, Option<String> rawLayoutO, Option<String> descriptionO,
                         Option<String> flagsO, AlbumSetAttrCallbacks callbacks)
    {
        final boolean changingTitle = titleO.exists(title -> !title.equals(album.getTitle()));
        final boolean changingCover = !rawCoverO.equals(album.getCoverId().map(ObjectId::toHexString));

        DjfsPrincipal principal = DjfsPrincipal.cons(uid);

        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            currentRevision += 1;

            Option<String> prevAlbumTitleO = titleO.isPresent() ? Option.of(album.getTitle()) : Option.empty();
            ListF<AlbumSetAttrUpdater> updaters = Cf.arrayList();
            updaters.addAll(setTitle(titleO));
            updaters.addAll(setLayout(rawLayoutO));
            updaters.addAll(setCover(uid, album, rawCoverO, coverOffsetYO));
            updaters.addAll(setCoverOffset(coverOffsetYO));
            updaters.addAll(setDescription(descriptionO));
            updaters.addAll(setFlags(getUniqFlagsList(flagsO)));
            if (changingTitle || changingCover) {
                updaters.addAll(setDateModified());
            }
            saveDeltasAndUpdateAlbumDB(album, currentRevision, updaters);

            AlbumSetAttrActivity albumSetAttrActivity = new AlbumSetAttrActivity(
                    principal, album, rawCoverO, coverOffsetYO, prevAlbumTitleO, titleO, Option.empty()
            );
            callbacks.callAfterAlbumSetAttrOutsideTransaction(albumSetAttrActivity);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);
            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            Album updatedAlbum = albumDao.findAlbum(uid, album.getId()).get();
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            if (titleO.isPresent()) {
                xivaPushGenerator.sendAlbumTitleChangedPush(uid, updatedAlbum);
            } else if (rawCoverO.isPresent()) {
                xivaPushGenerator.sendAlbumCoverChangedPush(uid, updatedAlbum);
            }
        }
    }

    public void removeAlbum(DjfsUid uid, Album album, AlbumRemoveCallbacks callbacks) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);
        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            AlbumRemoveActivity albumRemoveActivity = new AlbumRemoveActivity(principal, album);
            callbacks.callAfterAlbumRemoveOutsideTransaction(albumRemoveActivity);

            currentRevision += 1;
            albumDao.makeHidden(album);
            ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
            albumUpdateChanges.addAll(AlbumUtils.generateSetVisibilityChangeFields(false));

            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);

            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            xivaPushGenerator.sendAlbumRemovedPush(uid, album);
        }
    }

    public void mergeAlbums(DjfsUid uid, ObjectId srcId, ObjectId dstId) {
        MapF<ObjectId, Album> albumsById = getAlbumsWithTypeCheckSkipMissing(uid, Cf.list(srcId, dstId));
        if (!albumsById.containsKeyTs(srcId)) {
            return;
        }

        Album srcAlbum = albumsById.getTs(srcId);
        Album dstAlbum = albumsById.getTs(dstId);

        new MoveItemsOperation(uid, srcAlbum, dstAlbum)
                .andThen(new DeleteAlbumOperation(srcAlbum, false))
                .execute();
    }

    /**
        If indexing process was or wasn't started normally, it returns indexing state. On precondition fail, it throws.
    */
    public boolean startIndexing(DjfsUid uid, boolean resetAlbums) {
        final Tuple2<FacesIndexingState, Instant> oldState = userDao.getFacesIndexingState(uid);
        switch (oldState._1) {
            case REINDEXED:
                if (resetAlbums) {
                    logger.warn(
                            "Faces are already indexed, albums are created, and everything will be reset. uid = {}, oldState = {}",
                            uid,
                            oldState
                    );
                    break;
                } else {
                    logger.warn(
                            "Faces are already indexed, albums are created. Nothing to do. uid = {}, oldState = {}",
                            uid,
                            oldState
                    );
                    return false;
                }
            case RUNNING:
                if (resetAlbums) {
                    logger.warn(
                            "Face indexing is already running, and another process may be started. uid = {}, oldState = {}",
                            uid,
                            oldState
                    );
                    break;
                } else {
                    logger.warn(
                            "Face indexing is already running. Nothing to do. uid = {}, oldState = {}",
                            uid,
                            oldState
                    );
                    return false;
                }
            case NOT_INDEXED:
                userDao.findExistingAndNotBlocked(uid);
                final Option<String> userCountry = BlackboxUtils.getCountry(uid, blackbox);
                if (userCountry.exists(country -> !ALLOWED_COUNTRIES.containsTs(country.toUpperCase()))) {
                    userDao.setFacesIndexingState(uid, FacesIndexingState.COUNTRY_RESTRICTION);
                    throw new FeatureUnavailableInUserCountryException(uid, userCountry);
                }
                break;
            case USER_REFUSED:
                throw new UserRefusedException(uid);
            case COUNTRY_RESTRICTION:
                throw new FeatureUnavailableInUserCountryException(uid, Option.empty());
        }
        userDao.setFacesIndexingState(uid, FacesIndexingState.RUNNING);
        diskSearchHttpClient.reindexFaces(uid);
        return true;
    }

    private class MoveItemsOperation extends ConsistentOperation {
        private final Album srcAlbum;
        private final Album dstAlbum;

        public MoveItemsOperation(DjfsUid uid, Album srcAlbum, Album dstAlbum) {
            super(uid, AlbumType.FACES, Cf.set(srcAlbum, dstAlbum), true);
            this.srcAlbum = srcAlbum;
            this.dstAlbum = dstAlbum;
        }

        @Override
        protected OperationResult updateAndGetChanges() {

            ListF<AlbumItem> items = albumItemDao.getAllAlbumItems(uid, srcAlbum.getId());
            SetF<String> intersection = albumItemDao.getAllAlbumItems(uid, dstAlbum.getId())
                    .map(AlbumItem::getObjectId).unique()
                    .intersect(items.map(AlbumItem::getObjectId).unique());

            int dstItemsCount = albumItemDao.countObjectsInAlbum(uid, dstAlbum.getId());

            ListF<AlbumDeltaChange> insertChanges = items
                    .filterNot(item -> intersection.containsTs(item.getObjectId()))
                    .map(item -> item.withAlbumId(dstAlbum.getId()))
                    .map(AlbumUtils::createInsertChanges)
                    .plus1(AlbumUtils.createSetItemsCountChange(dstAlbum,
                            dstItemsCount + items.size() - intersection.size()));

            ListF<AlbumDeltaChange> deleteChanges = items.map(AlbumUtils::createDeleteChanges)
                    .plus1(AlbumUtils.createSetItemsCountChange(srcAlbum, 0));

            Option<AlbumDeltaChange> dstNameChange;
            Option<String> newDstTileO;
            if (dstAlbum.getTitle().isEmpty() && !srcAlbum.getTitle().isEmpty()) {
                String newDstTile = srcAlbum.getTitle();
                newDstTileO = Option.of(newDstTile);
                dstNameChange = Option.of(AlbumUtils.createSetTitleChange(dstAlbum, newDstTile));
                albumDao.setTitle(dstAlbum, newDstTile);
            } else {
                dstNameChange = Option.empty();
                newDstTileO = Option.empty();
            }

            albumItemDao.removeFromAlbum(uid, srcAlbum.getId(), intersection);
            albumItemDao.changeAlbumId(uid, srcAlbum.getId(), dstAlbum.getId());
            albumFaceClustersDao.changeAlbumId(uid, srcAlbum.getId(), dstAlbum.getId());

            return new OperationResult(insertChanges.plus(dstNameChange).plus(deleteChanges),
                    Cf.list(XivaPushGeneratorAlbumUtils.albumsMerged(uid, srcAlbum, dstAlbum, newDstTileO)));
        }
    }

    @Override
    protected Option<Double> itemOrderIndex(FileDjfsResource file) {
        final double time = file
                .getEtime()
                .getOrElse(() -> {
                    logger.info("No etime for file: " + file);
                    return InstantUtils.toSecondsLong(file.getCreationTime());
                });
        return Option.of(- time);
    }

    public void receiveUserRefusal(DjfsUid uid) {
        logger.info("User {} just opted out from face albums", uid);
        userDao.setFacesIndexingState(uid, FacesIndexingState.USER_REFUSED);
    }
}
