package ru.yandex.chemodan.app.djfs.core.album;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumSetAttrActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumsetattr.AlbumSetAttrCallbacks;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/** Tools for working with 'favorites' albums. */
public class FavoritesAlbumManager extends AbstractAlbumManager {
    private static final Logger logger = LoggerFactory.getLogger(FavoritesAlbumManager.class);

    public FavoritesAlbumManager(
            DjfsResourceDao djfsResourceDao,
            ShareInfoManager shareInfoManager,
            AlbumItemDao albumItemDao,
            AlbumDao albumDao,
            AlbumDeltaDao albumDeltaDao,
            XivaPushGenerator xivaPushGenerator,
            TransactionUtils transactionUtils
    ) {
        super(
                albumDao,
                albumItemDao,
                albumDeltaDao,
                transactionUtils,
                xivaPushGenerator,
                shareInfoManager,
                djfsResourceDao
        );
    }

    @Override
    public AlbumType getAlbumType() {
        return AlbumType.FAVORITES;
    }

    public void setAttrs(DjfsUid uid, Album album, Option<String> rawLayoutO, Option<String> descriptionO,
                         Option<String> flagsO, AlbumSetAttrCallbacks callbacks) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);

        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            currentRevision += 1;

            ListF<AlbumSetAttrUpdater> updaters = Cf.arrayList();
            updaters.addAll(setLayout(rawLayoutO));
            updaters.addAll(setDescription(descriptionO));
            updaters.addAll(setFlags(getUniqFlagsList(flagsO)));
            saveDeltasAndUpdateAlbumDB(album, currentRevision, updaters);

            AlbumSetAttrActivity albumSetAttrActivity = new AlbumSetAttrActivity(
                    principal, album, Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty()
            );
            callbacks.callAfterAlbumSetAttrOutsideTransaction(albumSetAttrActivity);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);
            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
        }
    }

    @Override
    protected Option<Double> itemOrderIndex(FileDjfsResource file) {
        final double time = file
                .getEtime()
                .getOrElse(() -> {
                    logger.info("No etime for file: " + file);
                    return InstantUtils.toSecondsLong(file.getCreationTime());
                });
        return Option.of(- time);
    }

}
