package ru.yandex.chemodan.app.djfs.core.album;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchHttpClient;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchResponse;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsAlbumsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class FetchAestheticsForMauFromIndexerTask extends YcridOnetimeTaskSupport<FetchAestheticsForMauFromIndexerTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(GeoAlbumResourceProcessingTask.class);

    private final DjfsResourceDao djfsResourceDao;
    private final DiskSearchHttpClient diskSearchHttpClient;
    private final UserDao userDao;

    public FetchAestheticsForMauFromIndexerTask(DjfsResourceDao djfsResourceDao,
            DiskSearchHttpClient diskSearchHttpClient, UserDao userDao)
    {
        super(Parameters.class);
        this.djfsResourceDao = djfsResourceDao;
        this.diskSearchHttpClient = diskSearchHttpClient;
        this.userDao = userDao;
    }

    public FetchAestheticsForMauFromIndexerTask(String uid)
    {
        super(new Parameters(uid));
        this.djfsResourceDao = null;
        this.diskSearchHttpClient = null;
        this.userDao = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context)
    {
        DjfsUid uid = DjfsUid.cons(parameters.uid, ActionContext.CLIENT_INPUT);

        UserData user;
        try {
            user = userDao.findExistingAndNotBlocked(uid);
        } catch (UserNotInitializedException | UserIsBlockedException e) {
            logger.info("User is blocked or not initialized: " + uid.asString());
            return;
        }

        long filesCount = djfsResourceDao.countAllFiles(uid);
        if (filesCount > 500000) {
            logger.info("User has too many files: uid=" + uid.asString() + ", count=" + filesCount);
            return;
        }

        final int totalLimit = 75000;  // hope that will be enough for all users
        ListF<DjfsFileId> fileIds = djfsResourceDao.getFileIdForFilesWithCoordinatesWithoutAesthetics(uid, totalLimit);

        int indexerBulkSize = 20;
        int databaseBulkWriteSize = 200;
        int startIndex = 0;
        ListF<DiskSearchHttpClient.FetchParameter> fields = Cf.list(
                DiskSearchHttpClient.FetchParameter.ID,
                DiskSearchHttpClient.FetchParameter.AESTHETICS);

        Tuple2List<DjfsFileId, Double> idWithAesthetics = Tuple2List.arrayList();
        while (startIndex < fileIds.size()) {
            ListF<DjfsFileId> chunk =
                    fileIds.subList(startIndex, Math.min(startIndex + indexerBulkSize, fileIds.size()));

            DiskSearchResponse data =
                    diskSearchHttpClient.getExtractedData(uid, chunk, fields, user.isQuickMoveUser());
            idWithAesthetics.addAll(
                    data.getHitsArray()
                            .filter(x -> x.getAesthetics().isPresent())
                            .map(x -> Tuple2.tuple(x.getFileId(), x.getAesthetics().get()))
            );

            while (idWithAesthetics.size() > databaseBulkWriteSize) {
                Tuple2List<DjfsFileId, Double> databaseChunk = idWithAesthetics.take(databaseBulkWriteSize);
                idWithAesthetics = idWithAesthetics.drop(databaseBulkWriteSize);

                djfsResourceDao.setAesthetics(uid, databaseChunk, true);
            }

            startIndex += indexerBulkSize;
        }

        if (idWithAesthetics.isNotEmpty()) {
            djfsResourceDao.setAesthetics(uid, idWithAesthetics, true);
        }

        if (fileIds.size() >= totalLimit) {
            throw new NotImplementedException("User " + uid.asString() + " has too many files with coordinates");
        }
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsAlbumsTaskQueueName.ALBUMS_INDEXER_TASKS;
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String uid;

        Parameters(String uid) {
            this.uid = uid;
        }
    }
}
