package ru.yandex.chemodan.app.djfs.core.album;

import javax.annotation.Nullable;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.collection.Tuple3List;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchHttpClient;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchResponse;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfoManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsAlbumsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class FetchCoordinatesFromIndexerTask extends YcridOnetimeTaskSupport<FetchCoordinatesFromIndexerTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(GeoAlbumResourceProcessingTask.class);

    private final DjfsResourceDao djfsResourceDao;
    private final DiskSearchHttpClient diskSearchHttpClient;
    private final UserDao userDao;
    private final DiskInfoManager diskInfoManager;
    private final BazingaTaskManager bazingaTaskManager;

    public FetchCoordinatesFromIndexerTask(DjfsResourceDao djfsResourceDao, DiskSearchHttpClient diskSearchHttpClient,
            UserDao userDao, DiskInfoManager diskInfoManager, BazingaTaskManager bazingaTaskManager)
    {
        super(Parameters.class);
        this.djfsResourceDao = djfsResourceDao;
        this.diskSearchHttpClient = diskSearchHttpClient;
        this.userDao = userDao;
        this.diskInfoManager = diskInfoManager;
        this.bazingaTaskManager = bazingaTaskManager;
    }

    public FetchCoordinatesFromIndexerTask(String uid)
    {
        this(uid, false);
    }

    public FetchCoordinatesFromIndexerTask(String uid, boolean submitAlbumGenerationTask)
    {
        super(new Parameters(uid, submitAlbumGenerationTask));
        this.djfsResourceDao = null;
        this.diskSearchHttpClient = null;
        this.userDao = null;
        this.diskInfoManager = null;
        this.bazingaTaskManager = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context)
    {
        DjfsUid uid = DjfsUid.cons(parameters.uid, ActionContext.CLIENT_INPUT);

        UserData user;
        try {
            user = userDao.findExistingAndNotBlocked(uid);
        } catch (UserNotInitializedException | UserIsBlockedException e) {
            logger.info("User is blocked or not initialized: " + uid.asString());
            return;
        }

        Option<DjfsFileId> startingFileId = Option.empty();
        int databaseBulkReadSize = 50000;
        boolean hasMoreData = true;

        while (hasMoreData) {
            ListF<DjfsFileId> fileIds = djfsResourceDao.getFileIds(uid, databaseBulkReadSize, startingFileId);

            int indexerBulkSize = 20;
            int databaseBulkWriteSize = 200;
            int startIndex = 0;
            ListF<DiskSearchHttpClient.FetchParameter> fields = Cf.list(
                    DiskSearchHttpClient.FetchParameter.ID,
                    DiskSearchHttpClient.FetchParameter.LATITUDE,
                    DiskSearchHttpClient.FetchParameter.LONGITUDE,
                    DiskSearchHttpClient.FetchParameter.AESTHETICS);

            Tuple3List<DjfsFileId, Coordinates, Option<Double>> idWithCoordinatesAndAesthetics = Tuple3List.arrayList();
            while (startIndex < fileIds.size()) {
                ListF<DjfsFileId> chunk =
                        fileIds.subList(startIndex, Math.min(startIndex + indexerBulkSize, fileIds.size()));

                DiskSearchResponse data =
                        diskSearchHttpClient.getExtractedData(uid, chunk, fields, user.isQuickMoveUser());
                idWithCoordinatesAndAesthetics.addAll(
                        data.getHitsArray()
                                .filter(x -> x.getCoordinates().isPresent())
                                .map(x -> Tuple3.tuple(x.getFileId(), x.getCoordinates().get(), x.getAesthetics()))
                );

                while (idWithCoordinatesAndAesthetics.size() > databaseBulkWriteSize) {
                    Tuple3List<DjfsFileId, Coordinates, Option<Double>> databaseChunk =
                            idWithCoordinatesAndAesthetics.take(databaseBulkWriteSize);
                    idWithCoordinatesAndAesthetics = idWithCoordinatesAndAesthetics.drop(databaseBulkWriteSize);

                    djfsResourceDao.setCoordinates(uid, databaseChunk.get12(), true);

                    Tuple2List<DjfsFileId, Double> filesWithAesthetics =
                            databaseChunk.get13().filterBy2(Option::isPresent).map2(Option::get);
                    if (filesWithAesthetics.isNotEmpty()) {
                        djfsResourceDao.setAesthetics(uid, filesWithAesthetics, true);
                    }
                }

                startIndex += indexerBulkSize;
            }

            if (idWithCoordinatesAndAesthetics.isNotEmpty()) {
                djfsResourceDao.setCoordinates(uid, idWithCoordinatesAndAesthetics.get12(), true);

                Tuple2List<DjfsFileId, Double> filesWithAesthetics =
                        idWithCoordinatesAndAesthetics.get13().filterBy2(Option::isPresent).map2(Option::get);
                if (filesWithAesthetics.isNotEmpty()) {
                    djfsResourceDao.setAesthetics(uid, filesWithAesthetics, true);
                }
            }

            if (startingFileId.isPresent() && startingFileId.get().equals(fileIds.last())) {
                throw new NotImplementedException("User " + uid.asString() + " has too many file id doubles");
            }
            hasMoreData = fileIds.size() == databaseBulkReadSize;
            if (hasMoreData) {
                startingFileId = Option.of(fileIds.last());
            }
        }

        diskInfoManager.setIndexerSyncGeoTimestamp(uid, Instant.now());

        if (parameters.submitAlbumGenerationTask && !diskInfoManager.isGeoAlbumsGeneratingInProgress(uid)) {
            try {
                bazingaTaskManager.schedule(new GenerateGeoAlbumsTask(uid.asLong()));
            } catch (Exception e) {  // do nothing here, will be submitted later in snapshot handler
                logger.error("Generate geo album task submit is failed", e);
            }
        }
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsAlbumsTaskQueueName.ALBUMS_INDEXER_TASKS;
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return FetchCoordinatesFromIndexerTask.Parameters.Converter.class;
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String uid;
        private final boolean submitAlbumGenerationTask;

        Parameters(String uid) {
            this(uid, false);
        }

        Parameters(String uid, boolean submitAlbumGenerationTask) {
            this.uid = uid;
            this.submitAlbumGenerationTask = submitAlbumGenerationTask;
        }

        public static class Converter implements ActiveUniqueIdentifierConverter<
            FetchCoordinatesFromIndexerTask.Parameters, FetchCoordinatesFromIndexerTask.Parameters>
        {
            @Override
            public Class<FetchCoordinatesFromIndexerTask.Parameters> getActiveUniqueIdentifierClass() {
                return FetchCoordinatesFromIndexerTask.Parameters.class;
            }

            @Override
            public FetchCoordinatesFromIndexerTask.Parameters convert(
                FetchCoordinatesFromIndexerTask.Parameters parameters)
            {
                return parameters;
            }
        }
    }
}
