package ru.yandex.chemodan.app.djfs.core.album;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsTaskManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsCeleryOnetimeTask;
import ru.yandex.chemodan.queller.worker.CeleryOnetimeTaskParameters;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;


@RequiredArgsConstructor
public class FileOperationPostProcessHandler extends DjfsCeleryOnetimeTask {

    public static final TaskId TASK_ID = new TaskId("mpfs.core.albums.job_handlers.handle_file_changed_operation");

    private final DjfsAlbumsTaskManager djfsAlbumsTaskManager;
    private final GeoAlbumManager geoAlbumManager;
    private final PersonalAlbumManager personalAlbumManager;

    private final DynamicProperty<Boolean> enablePersonalCleaning = new DynamicProperty<>(
            "disk-djfs-albums-enable-clean-personal", false);
    private final DynamicProperty<Boolean> disableLinearization = new DynamicProperty<>(
            "disk-djfs-albums-disable-linearization", true);

    @Override
    protected void handle(CeleryOnetimeTaskParameters parameters, ExecutionContext context) {
        JsonArray resourceIdsArray = (JsonArray) (parameters.kwargs.getO("resource_ids").get());
        ListF<JsonValue> resourceIdsValues = Cf.toList(resourceIdsArray.getArray()).cast();
        ListF<DjfsResourceId> resourceIds =
                resourceIdsValues.cast(JsonString.class).map(JsonString::getString).map(DjfsResourceId::cons);
        if (disableLinearization.get()) {
            geoAlbumManager.postProcessFiles(resourceIds);
            if (enablePersonalCleaning.get()) {
                personalAlbumManager.postProcessFiles(resourceIds);
            }
        } else {
            for (DjfsResourceId resourceId : resourceIds) {
                djfsAlbumsTaskManager.schedule(
                        new FileOperationPostProcessingTask(resourceId.getUid().asString(), resourceId.getValue())
                );
            }
        }
    }

    @Override
    protected TaskId celeryTaskId() {
        return TASK_ID;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(32);
    }
}
