package ru.yandex.chemodan.app.djfs.core.album;

import com.mongodb.ReadPreference;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsAlbumsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public abstract class GeoAlbumBaseProcessingTask extends AlbumBaseProcessingTask {
    private static final Logger logger = LoggerFactory.getLogger(GeoAlbumBaseProcessingTask.class);

    protected final GeoAlbumManager geoAlbumManager;
    protected final Filesystem filesystem;
    protected final TransactionUtils transactionUtils;

    public GeoAlbumBaseProcessingTask(Filesystem filesystem, GeoAlbumManager geoAlbumManager,
            TransactionUtils transactionUtils)
    {
        super();
        this.geoAlbumManager = geoAlbumManager;
        this.filesystem = filesystem;
        this.transactionUtils = transactionUtils;
    }

    public GeoAlbumBaseProcessingTask(String uid, String resourceId) {
        super(uid, resourceId);
        this.geoAlbumManager = null;
        this.filesystem = null;
        this.transactionUtils = null;
    }

    @Override
    public void doExecute(Parameters parameters, ExecutionContext context) {
        DjfsUid uid = DjfsUid.cons(parameters.getUid(), ActionContext.CLIENT_INPUT);
        DjfsResourceId resourceId = DjfsResourceId.cons(parameters.getResourceId(), ActionContext.CLIENT_INPUT);

        if (!uid.equals(resourceId.getUid())) {
            logger.info("Geo album generation is asked for file with different uid (user: " + uid.asString() +
                    ", resource_id: " + resourceId.getValue() + ")");
            return;
        }

        try {
            Option<DjfsResource> resource = filesystem.find(DjfsPrincipal.cons(uid), resourceId, Option.of(ReadPreference.primary())).firstO();
            if (!resource.isPresent() || !(resource.get() instanceof FileDjfsResource)) {
                logger.info("Resource is not found or not a file (" + resourceId.getValue() + ")");
                return;
            }
            FileDjfsResource file = (FileDjfsResource) resource.get();

            ListF<DefaultCityStrategy> strategies = geoAlbumManager.getGenerationStrategies(uid);
            Language locale = geoAlbumManager.getLocale(uid);

            transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
                geoAlbumManager.getAndLockCurrentRevision(uid);

                ListF<GeoInfo> geoInfos = geoAlbumManager.getResourceInfos(strategies, file, locale);
                if (geoInfos.isEmpty()) {
                    logger.info("Geo info is not found for resource " + resourceId.getValue());
                    return;
                }
                geoInfos.forEach(i -> processResource(file, i));
            });
        } catch (UserIsBlockedException e) {
            // do nothing in case of blocked user
        }
    }

    protected abstract void processResource(FileDjfsResource file, GeoInfo geoInfo);

    @Override
    public TaskQueueName queueName() {
        return DjfsAlbumsTaskQueueName.ALBUMS_GEO_TASKS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(100);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new CompoundReschedulePolicy(
                new RescheduleConstant(Duration.standardMinutes(1), 10),
                new RescheduleExponential(Duration.standardMinutes(1), 10)
        );
    }
}
