package ru.yandex.chemodan.app.djfs.core.album;

import lombok.Data;
import org.bson.types.ObjectId;
import org.joda.time.Instant;
import org.junit.Assume;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.album.pojo.AlbumAddItemResultPojo;
import ru.yandex.chemodan.app.djfs.core.client.ProfileAddressType;
import ru.yandex.chemodan.app.djfs.core.db.pg.TransactionUtils;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfoManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumSetAttrActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.albumsetattr.AlbumSetAttrCallbacks;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserBlockedException;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserLocale;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.geobase.Geobase;
import ru.yandex.inside.geobase.LinguisticsItem;
import ru.yandex.inside.geobase.RegionNode;
import ru.yandex.inside.geobase.RegionType;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class GeoAlbumManager extends AbstractAlbumManager {
    private static final Logger logger = LoggerFactory.getLogger(GeoAlbumManager.class);
    private static final ListF<ProfileAddressType> DETAILED_ADDRESS_TYPES = Cf.list(ProfileAddressType.HOME, ProfileAddressType.WORK);
    private static final String GEO_ALBUM_TYPE_EXCLUSION = "geo";

    private final Geobase geobase;
    private final UserDao userDao;
    private final DiskInfoManager diskInfoManager;
    private final GeoStrategyProvider geoStrategyProvider;
    private final BazingaTaskManager bazingaTaskManager;

    private final DynamicProperty<Boolean> doNotSaveCreatedAlbums = new DynamicProperty<>("disk-djfs-albums-generate-geo-albums-not-save", false);
    private final DynamicProperty<Long> filesMaxCountLimit = new DynamicProperty<>("disk-djfs-albums-generate-geo-albums-files-max-count", 500000L);

    private final Language defaultLocale = Language.RUSSIAN;
    private final int visibilityThreshold = 30;

    private final ListF<RegionType> federalSubjectChildTypes = Cf.list(RegionType.CITY, RegionType.VILLAGE,
            RegionType.SETTLEMENT);

    private final MapF<Integer, MapF<Language, String>> customRegionNames = Cf.map(
            1, Cf.map(
                    Language.RUSSIAN, "Московская область",
                    Language.ENGLISH, "Moscow region",
                    Language.UKRAINIAN, "Московська область",
                    Language.TURKISH, "Moskova Oblastı"
            ),
            10174, Cf.map(
                    Language.RUSSIAN, "Ленинградская область",
                    Language.ENGLISH, "Leningrad region",
                    Language.UKRAINIAN, "Ленінградська область",
                    Language.TURKISH, "Leningradskaya Oblastı"
            )
    );

    private final MapF<Language, Language> fallbackLanguages = Cf.map(
            Language.UKRAINIAN, Language.RUSSIAN,
            Language.TURKISH, Language.ENGLISH
    );

    protected GeoAlbumManager(AlbumDao albumDao, AlbumItemDao albumItemDao, Geobase geobase,
                              AlbumDeltaDao albumDeltaDao, TransactionUtils transactionUtils,
                              XivaPushGenerator xivaPushGenerator, DjfsResourceDao djfsResourceDao, UserDao userDao,
                              DiskInfoManager diskInfoManager, GeoStrategyProvider geoStrategyProvider,
                              BazingaTaskManager bazingaTaskManager,
                              ShareInfoManager shareInfoManager
    ) {
        super(
                albumDao,
                albumItemDao,
                albumDeltaDao,
                transactionUtils,
                xivaPushGenerator,
                shareInfoManager,
                djfsResourceDao
        );
        this.geobase = geobase;
        this.userDao = userDao;
        this.diskInfoManager = diskInfoManager;
        this.geoStrategyProvider = geoStrategyProvider;
        this.bazingaTaskManager = bazingaTaskManager;
    }

    @Override
    public AlbumType getAlbumType() {
        return AlbumType.GEO;
    }

    public ListF<DefaultCityStrategy> getGenerationStrategies(DjfsUid uid) {
        return geoStrategyProvider.get(uid);
    }

    public ListF<DefaultCityStrategy> getRemoveStrategies() {
        return Cf.list(geoStrategyProvider.allAlbumsStrategy());
    }

    public ListF<GeoInfo> getResourceInfos(ListF<DefaultCityStrategy> strategies, FileDjfsResource resource,
                                           Language locale)
    {
        return getResourceGeoIds(strategies, resource).map(x -> makeGeoInfo(x, locale));
    }

    public CollectionF<Integer> getResourceGeoIds(ListF<DefaultCityStrategy> strategies, FileDjfsResource resource)
    {
        MapF<RegionType, Integer> allResourceGeoIds = getAllResourceGeoIds(strategies, resource);

        for (RegionType type : federalSubjectChildTypes) {
            if (allResourceGeoIds.containsKeyTs(type)) {
                Integer geoId = allResourceGeoIds.getOrThrow(type);
                Option<Album> geoAlbumO = albumDao.findGeoAlbum(resource.getUid(), geoId);
                if (geoAlbumO.isPresent() && !geoAlbumO.get().isHidden()) {
                    allResourceGeoIds.removeO(RegionType.FEDERAL_SUBJECT);
                    break;
                }
            }
        }

        Tuple2List<RegionType, Integer> geoIdSortedByRegionType =
                allResourceGeoIds.entries().sortedBy(x -> -x._1.value());  // sort from small to big areas, it's important

        return geoIdSortedByRegionType.get2();
    }

    public MapF<RegionType, Integer> getAllResourceGeoIds(ListF<DefaultCityStrategy> strategies,
                                                          FileDjfsResource resource)
    {
        MapF<RegionType, Integer> result = Cf.hashMap();
        for (DefaultCityStrategy strategy : strategies) {
            result.putAll(strategy.get(resource));
        }
        return result;
    }

    private GeoInfo makeGeoInfo(int regionId, Language language) {
        if (!geobase.getSupportedLinguistics().containsTs(language.value())) {
            language = defaultLocale;
        }

        String regionName;
        Option<MapF<Language, String>> regionNameO = customRegionNames.getO(regionId);

        if (regionNameO.isPresent() && regionNameO.get().getO(language).isPresent()) {
            regionName = regionNameO.get().getOrThrow(language);
        } else {
            Option<LinguisticsItem> lingItem = geobase.getLinguisticsItemByRegionId(regionId, language.value());
            regionName = lingItem.get().getNominativeCase();

            // some locales have empty strings and we need to fallback
            if (regionName.isEmpty() && fallbackLanguages.getO(language).isPresent()) {
                Language fallbackLanguage = fallbackLanguages.getOrThrow(language);
                lingItem = geobase.getLinguisticsItemByRegionId(regionId, fallbackLanguage.value());
                regionName = lingItem.get().getNominativeCase();
            }
        }
        RegionType type = geobase.getRegionById(regionId).get().getType();

        return new GeoInfo(regionId, regionName, type);
    }

    public void setAttrs(DjfsUid uid, Album album, Option<String> rawCoverO,
                         Option<Double> coverOffsetYO, Option<String> rawLayoutO, Option<String> descriptionO,
                         Option<String> flagsO, AlbumSetAttrCallbacks callbacks) {
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);

        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            currentRevision += 1;

            ListF<AlbumSetAttrUpdater> updaters = Cf.arrayList();
            updaters.addAll(setLayout(rawLayoutO));
            updaters.addAll(setCover(uid, album, rawCoverO, coverOffsetYO));
            updaters.addAll(setCoverOffset(coverOffsetYO));
            updaters.addAll(setDescription(descriptionO));
            updaters.addAll(setFlags(getUniqFlagsList(flagsO)));
            if (!rawCoverO.equals(album.getCoverId())) {
                updaters.addAll(setDateModified());
            }

            saveDeltasAndUpdateAlbumDB(album, currentRevision, updaters);

            AlbumSetAttrActivity albumSetAttrActivity = new AlbumSetAttrActivity(
                    principal, album, rawCoverO, coverOffsetYO, Option.empty(), Option.empty(), Option.empty()
            );
            callbacks.callAfterAlbumSetAttrOutsideTransaction (albumSetAttrActivity);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.updateCurrentRevision(uid, currentRevision);
            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            if (rawCoverO.isPresent()) {
                xivaPushGenerator.sendAlbumCoverChangedPush(uid, albumDao.findAlbum(uid, album.getId()).get());
            }
        }
    }

    public void removeFromGeoAlbum(FileDjfsResource resource, int geoId) {
        removeFromGeoAlbum(resource.getResourceId().get(), geoId);
    }

    private void removeFromGeoAlbum(DjfsResourceId resourceId, int geoId) {
        DjfsUid uid = resourceId.getUid();

        Option<Album> albumO = albumDao.findGeoAlbum(resourceId.getUid(), geoId);
        if (!albumO.isPresent()) {
            logger.info("Album is not found for file " + resourceId.getValue());
            return;
        }

        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);
            Album album = albumO.get();

            String resourceFileId = resourceId.getFileId().getValue();
            ListF<AlbumItem> items = albumItemDao
                    .findObjectInAlbum(resourceId.getUid(), albumO.get().getId(), resourceFileId);
            if (items.isEmpty()) {
                logger.info("File " + resourceId.getValue() + " not found in album");
                return Tuple2.tuple(false, 0L);
            }
            albumItemDao.removeFromAlbum(resourceId.getUid(), albumO.get().getId(), resourceFileId);
            AlbumItem albumItem = items.first();

            currentRevision += 1;
            AlbumDelta albumItemDelta = AlbumUtils.createDeleteAlbumItemDelta(album, albumItem, currentRevision);
            albumDeltaDao.insert(albumItemDelta);

            ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
            int itemsCount = albumItemDao.countObjectsInAlbum(uid, album.getId());
            albumUpdateChanges.addAll(AlbumUtils.generateSetItemsCountChangeFields(itemsCount));

            if (album.getCoverId().isPresent() && albumItem.getId().equals(album.getCoverId().get())) {
                ListF<AlbumDeltaFieldChange> updateCoverDelta = selectNewCover(album);
                albumUpdateChanges.addAll(updateCoverDelta);
            }

            if (itemsCount == 0) {
                albumDao.makeHidden(album);
                albumUpdateChanges.addAll(AlbumUtils.generateSetVisibilityChangeFields(false));
            }

            currentRevision += 1;
            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);

            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
        }
    }

    public MapF<Album, ListF<AlbumItem>> generateGeoAlbumsForUser(DjfsUid uid) {
        UserData user;
        try {
            user = userDao.findExistingAndNotBlocked(uid);
        } catch (UserBlockedException | UserNotInitializedException e) {
            logger.info("User {} is blocked or not exists", uid.asString());
            return Cf.map();
        }
        Option<Long> currentRevisionO = albumDeltaDao.getCurrentRevisionWithoutLock(uid);
        if (currentRevisionO.map(revision -> revision > 0L).getOrElse(Boolean.FALSE)) {
            logger.debug("Albums has been generated already");
            return Cf.map();
        }
        long filesCount = djfsResourceDao.countAllFiles(uid);
        if (filesCount > filesMaxCountLimit.get()) {
            logger.warn("Too many files for geo albums generation uid={} count={}", uid, filesCount);
            diskInfoManager.disableGeoAlbumsGenerationInProgress(uid);
            return Cf.map();
        }
        diskInfoManager.enableGeoAlbumsGenerationInProgress(uid);
        ListF<FileDjfsResource> allPhotos = djfsResourceDao.getUserPhotosWithGeoCoordinates(user.getId())
                .filter(photo -> photo.getEtime().isPresent())
                .filter(photo -> photo.getPath().getArea() == DjfsResourceArea.DISK || photo.getPath().getArea() == DjfsResourceArea.PHOTOUNLIM)
                .filter(photo -> !photo.getAlbumsExclusions().getOrElse(Cf::list).containsTs(GEO_ALBUM_TYPE_EXCLUSION));
        SetF<CityUnderSubject> cityUnderSubjects = Cf.hashSet();
        MapF<Integer, ListF<FileDjfsResource>> photosByGeoIds = groupByCitiesAndDistricts(allPhotos, cityUnderSubjects,
                geoStrategyProvider.get(uid));
        filterFederalSubjectAlbums(photosByGeoIds, cityUnderSubjects);
        MapF<Album, ListF<AlbumItem>> albums = Cf.hashMap();
        MapF<Album, String> albumCovers = Cf.hashMap();
        photosByGeoIds
                .filterValues(ListF::isNotEmpty)
                .forEach((regionId, photos) -> addAlbumDataToAccumulator(regionId, user, photos, albums, albumCovers));
        if (doNotSaveCreatedAlbums.get()) {
            return Cf.x(albums);
        }
        transactionUtils.executeInNewOrCurrentTransaction(uid, () -> createAlbumsInDb(albums, albumCovers, uid));
        return Cf.x(albums);
    }

    private void filterFederalSubjectAlbums(MapF<Integer, ListF<FileDjfsResource>> allAlbums,
                                            SetF<CityUnderSubject> cityUnderSubjects)
    {
        for (CityUnderSubject cityUnderSubject : cityUnderSubjects) {
            ListF<FileDjfsResource> cityAlbumPhotos = allAlbums.getTs(cityUnderSubject.getCityGeoId());
            if (isHidden(cityAlbumPhotos)) {
                continue;
            }
            ListF<FileDjfsResource> fsAlbumPhotos = Cf.toArrayList(allAlbums.getTs(cityUnderSubject.getSubjectGeoId()));
            fsAlbumPhotos.removeAllTs(cityAlbumPhotos);
            allAlbums.put(cityUnderSubject.getSubjectGeoId(), fsAlbumPhotos.unmodifiable());
        }
    }

    private void createAlbumsInDb(MapF<Album, ListF<AlbumItem>> albums, MapF<Album, String> albumCovers, DjfsUid uid) {
        long currentRevision = getAndLockCurrentRevision(uid);
        if (currentRevision > 0) {
            logger.debug("Current revision is bigger than 0. Album has been already generated uid={} currentRevision={}", uid, currentRevision);
            diskInfoManager.disableGeoAlbumsGenerationInProgress(uid);
            return;
        }
        removeAllExistingGeoAlbums(uid);
        albums.forEach((album, albumItems) -> saveAlbum(album, albumItems, albumCovers));
        commitAlbumsCreation(albums.keys(), uid, currentRevision + 1);
        diskInfoManager.disableGeoAlbumsGenerationInProgress(uid);
    }

    private void commitAlbumsCreation(ListF<Album> albums, DjfsUid uid, long revision) {
        albumDeltaDao.tryInitializeCurrentRevision(uid);
        albumDeltaDao.updateCurrentRevision(uid, revision);
        if (albums.isEmpty()) {
            return;
        }
        albumDao.updateAlbumsRevision(albums, uid, revision);
    }

    private void removeAllExistingGeoAlbums(DjfsUid uid) {
        ListF<Album> albums = albumDao.getAlbums(uid, AlbumType.GEO);
        albums.forEach(album -> albumItemDao.removeAllItemsFromAlbum(uid, album.getId()));
        albums.forEach(albumDao::delete);
    }

    private MapF<Integer, ListF<FileDjfsResource>> groupByCitiesAndDistricts(ListF<FileDjfsResource> allPhotos,
                                                                             SetF<CityUnderSubject> cityUnderSubjectAccumulator,
                                                                             ListF<DefaultCityStrategy> cityStrategies) {
        MapF<Integer, ListF<FileDjfsResource>> result = Cf.hashMap();
        allPhotos.forEach(photo -> addPhotoToCityAndDistrict(photo, result, cityUnderSubjectAccumulator, cityStrategies));
        return result.mapValues(this::filterAlbumPhotosIfNeeded);
    }

    private void addPhotoToCityAndDistrict(FileDjfsResource photo,
                                           MapF<Integer, ListF<FileDjfsResource>> accumulator, SetF<CityUnderSubject> cityUnderSubjectAccumulator,
                                           ListF<DefaultCityStrategy> cityStrategies)
    {
        MapF<RegionType, Integer> geoIds = getAllResourceGeoIds(cityStrategies, photo);
        geoIds.getO(RegionType.CITY).ifPresent(cityGeoId ->
                geoIds.getO(RegionType.FEDERAL_SUBJECT)
                        .ifPresent(fsGeoId -> cityUnderSubjectAccumulator.add(new CityUnderSubject(fsGeoId, cityGeoId))));
        geoIds.values().forEach(id -> addPhotoToRegionId(accumulator, id, photo));
    }

    private void addPhotoToRegionId(MapF<Integer, ListF<FileDjfsResource>> accumulator, int regionId,
                                    FileDjfsResource photo)
    {
        ListF<FileDjfsResource> photosForRegion = accumulator.getO(regionId).getOrElse(Cf::arrayList);
        photosForRegion.add(photo);
        accumulator.put(regionId, photosForRegion);
    }

    private ListF<FileDjfsResource> filterAlbumPhotosIfNeeded(ListF<FileDjfsResource> photos) {
        if (photos.size() <= AlbumUtils.GEO_ALBUM_SIZE_LIMIT) {
            return photos;
        }
        return photos.sortedByDesc(photo -> photo.getExifTime().map(Instant::getMillis).getOrElse(Long.MIN_VALUE))
                .take(AlbumUtils.GEO_ALBUM_SIZE_LIMIT);
    }

    private Album createAlbum(Integer regionId, UserData user, ListF<FileDjfsResource> photos) {
        Instant now = Instant.now();

        Language userLanguage = user.getLocale().map(UserLocale::toLanguage).getOrElse(defaultLocale);
        String name = makeGeoInfo(regionId, userLanguage).regionName;

        return Album.builder()
                .id(new ObjectId())
                .uid(user.getId())
                .type(AlbumType.GEO)
                .title(name)
                .description(Option.empty())
                .geoId(Option.of((long) regionId))
                .revision(Option.empty())
                .hidden(isHidden(photos))
                .dateCreated(Option.of(now))
                .dateModified(photos.maxByO(photo -> photo.getExifTime().map(Instant::getMillis).getOrElse(0L)).flatMapO(FileDjfsResource::getExifTime))
                .coverId(Option.empty())
                .coverOffsetY(Option.empty())
                .flags(Option.empty())
                .layout(Option.empty())
                .isBlocked(false)
                .blockReason(Option.empty())
                .fotkiAlbumId(Option.empty())
                .isPublic(false)
                .publicKey(Option.empty())
                .publicUrl(Option.empty())
                .shortUrl(Option.empty())
                .socialCoverStid(Option.empty())
                .isDescSorting(Option.empty())
                .albumItemsSorting(Option.empty())
                .build();
    }

    private boolean isHidden(ListF<FileDjfsResource> allPhotosOfAlbum) {
        return allPhotosOfAlbum.size() < visibilityThreshold;
    }

    private AlbumItem createAlbumItem(Album album, FileDjfsResource photo) {
        return AlbumItem.builder()
                .id(new ObjectId())
                .uid(album.getUid())
                .description(Option.empty())
                .groupId(Option.empty())
                .objectId(photo.getResourceId().map(DjfsResourceId::getFileId).map(DjfsFileId::getValue).getOrThrow("File id is empty"))
                .objectType(AlbumItemType.RESOURCE)
                .orderIndex(Option.of(-(double)(photo.getEtime().getOrThrow("Etime is empty"))))
                .albumId(album.getId())
                .faceInfo(Option.empty())
                .dateCreated(Option.of(Instant.now()))
                .build();
    }

    private void saveAlbum(Album album, ListF<AlbumItem> items, MapF<Album, String> covers) {
        ObjectId coverAlbumItemId = items.find(photo -> covers.getTs(album).equals(photo.getObjectId()))
                .getOrThrow(String.format("Cannot find a cover for album %s", album.getTitle())).getId();;
        albumDao.insert(album.withRevision(0L));
        albumItemDao.batchInsert(album.getUid(), items);
        albumDao.setCover(album, coverAlbumItemId);
    }

    private void addAlbumDataToAccumulator(Integer regionId, UserData user, ListF<FileDjfsResource> photos,
                                           MapF<Album, ListF<AlbumItem>> albumAccumulator, MapF<Album, String> albumCoverAccumulator)
    {
        Album album = createAlbum(regionId, user, photos);
        albumAccumulator.put(album, photos.map(photo -> createAlbumItem(album, photo)));
        albumCoverAccumulator.put(album, photos.maxByO(photo -> photo.getAesthetics().getOrElse(Double.NEGATIVE_INFINITY))
                .getOrElse(photos::first).getResourceId().get().getFileId().getValue());
    }

    private ListF<AlbumDeltaFieldChange> selectNewCover(Album album) {
        ListF<AlbumItem> allItems = albumItemDao.getAllAlbumItems(
                album.getUid(), Cf.list(album.getId()), AlbumUtils.GEO_ALBUM_SIZE_LIMIT);

        if (allItems.isEmpty()) {
            return Cf.list();  // TODO: generate cover remove delta and sunset cover id for album???
        }

        Tuple2<ObjectId, DjfsResourceId> newCover = findNewCover(album.getUid(), allItems);
        albumDao.setCover(album, newCover._1);
        return AlbumUtils.generateSetCoverChangeFields(newCover._1, newCover._2);
    }

    private Tuple2<ObjectId, DjfsResourceId> findNewCover(DjfsUid uid, ListF<AlbumItem> albumItems) {
        Assume.assumeTrue(albumItems.isNotEmpty());

        int bulkSize = 1000;
        int startIndex = 0;

        ObjectId mostBeautifulItemId = albumItems.first().getId();
        DjfsFileId mostBeautifulItemFileId = DjfsFileId.cons(albumItems.first().getObjectId());
        Double maxAesthetics = djfsResourceDao.getAesthetics(uid, Cf.list(mostBeautifulItemFileId))
                .map(x -> x._2.getOrElse(0d)).firstO().getOrElse(0d);

        while (startIndex < albumItems.size()) {
            MapF<DjfsFileId, ObjectId> fileIdToItemId =
                    albumItems.subList(startIndex, Math.min(albumItems.size(), startIndex + bulkSize))
                            .toMap(x -> Tuple2.tuple(DjfsFileId.cons(x.getObjectId()), x.getId()));

            if (fileIdToItemId.isEmpty()) {
                break;
            }

            ListF<Tuple2<DjfsFileId, Option<Double>>> objIdWithAesthetics = djfsResourceDao.getAesthetics(uid,
                    fileIdToItemId.keys());

            for (Tuple2<DjfsFileId, Option<Double>> entry : objIdWithAesthetics) {
                if (entry._2.isPresent() && entry._2.get() > maxAesthetics) {
                    mostBeautifulItemFileId = entry._1;
                    maxAesthetics = entry._2.get();
                    mostBeautifulItemId = fileIdToItemId.getTs(mostBeautifulItemFileId);
                }
            }

            startIndex += bulkSize;
        }

        return Tuple2.tuple(mostBeautifulItemId, DjfsResourceId.cons(uid, mostBeautifulItemFileId));
    }

    public void addPhotoToAlbum(FileDjfsResource resource, GeoInfo info) {
        addPhotoToAlbum(resource, info, Option.empty());
    }

    public void addPhotoToAlbum(FileDjfsResource resource, GeoInfo info, Option<Integer> childGeoId) {
        if (resource.getAlbumsExclusions().isPresent() &&
                resource.getAlbumsExclusions().get().containsTs(GEO_ALBUM_TYPE_EXCLUSION))
        {
            logger.info("File is excluded from geo: " + resource.getUid() + ", " + resource.getPath());
            return;
        }
        if (!resource.getEtime().isPresent() || !resource.getResourceId().isPresent()) {
            logger.info("No etime or file id is found for file: " + resource.getUid() + ", " + resource.getPath());
            return;
        }
        Long resourceEtime = resource.getEtime().get();
        Instant resourceExifTime = resource.getExifTime().get();
        Instant newAlbumModificationDate = resourceExifTime.isAfter(Instant.now()) ? Instant.now() : resourceExifTime;

        DjfsResourceId fileResourceId = resource.getResourceId().get();
        DjfsUid uid = resource.getUid();
        Option<Long> currentRevisionO = albumDeltaDao.getCurrentRevisionWithoutLock(uid);
        if (isGeoAlbumsGeneratingInProgress(uid)) {
            throw new RuntimeException("Albums generating is in progress");
        }
        if (!currentRevisionO.isPresent()) {
            logger.debug("Albums generating has not been started");
            return;
        }

        Option<AlbumAddItemResultPojo> addItemResultO = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            Option<ExtendedAlbum> albumO = albumDao.findExtendedGeoAlbum(uid, info.regionId);

            if (albumO.isPresent()) {
                ObjectId albumId = albumO.get().album.getId();
                ListF<AlbumItem> existingItem = albumItemDao.findObjectInAlbum(uid, albumId,
                        fileResourceId.getFileId().getValue());
                if (existingItem.isNotEmpty()) {
                    logger.info(
                            "Item " + resource.getResourceId() + " is already in the album " + albumId.toHexString() +
                                    ", adding is skipped");
                    return Option.empty();
                }
            }

            if (checkChildAlbumVisibilityForFederalSubjectAlbum(uid, info, childGeoId)) {
                return Option.empty();
            }

            Album album = albumO.map(x -> x.album).getOrNull();
            boolean isNewAlbum = false;

            int itemsCountBefore = 0;
            if (!albumO.isPresent()) {
                album = prepareAlbum(uid, info.regionId, info.regionName, currentRevision, resourceExifTime);
                albumDao.insert(album);

                currentRevision += 1;
                AlbumDelta insertAlbumDelta =
                        AlbumUtils.createInsertAlbumDelta(album, 1, resource.getResourceId(), currentRevision);
                albumDeltaDao.insert(insertAlbumDelta);

                isNewAlbum = true;
            } else {
                ObjectId albumId = albumO.get().album.getId();
                itemsCountBefore = albumItemDao.countObjectsInAlbum(uid, albumId);
            }

            AlbumItem item = prepareAlbumItem(album, fileResourceId.getFileId(), resourceEtime);
            albumItemDao.insert(item);

            currentRevision += 1;
            AlbumDelta albumItemDelta = AlbumUtils.createInsertAlbumItemDelta(album, item, currentRevision);
            albumDeltaDao.insert(albumItemDelta);

            ListF<AlbumDeltaFieldChange> albumUpdateChanges = Cf.arrayList();
            albumUpdateChanges.addAll(AlbumUtils.generateSetItemsCountChangeFields(itemsCountBefore + 1));

            if (isNewAlbum) {
                albumDao.setCover(album, item.getId());
                albumUpdateChanges.addAll(AlbumUtils.generateSetCoverChangeFields(item.getId(), fileResourceId));
            } else if (resource.getAesthetics().isPresent()) {
                Double resourceAesthetics = resource.getAesthetics().get();
                if (shouldUpdateCover(resourceAesthetics, albumO.get())) {
                    albumDao.setCover(album, item.getId());
                    albumUpdateChanges.addAll(AlbumUtils.generateSetCoverChangeFields(item.getId(), fileResourceId));
                }
            }

            boolean albumChangedVisibility = false;
            if (itemsCountBefore == visibilityThreshold - 1 && album.isHidden()) {
                albumDao.makeVisible(album);
                albumUpdateChanges.addAll(AlbumUtils.generateSetVisibilityChangeFields(true));
                albumChangedVisibility = true;
            }

            if (!album.getDateModified().isPresent() ||
                    album.getDateModified().get().isBefore(newAlbumModificationDate))
            {
                albumDao.updateDateModified(album, newAlbumModificationDate);
                albumUpdateChanges.addAll(AlbumUtils.generateSetDateModifyChangeFields(newAlbumModificationDate));
            }

            currentRevision += 1;
            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges, currentRevision);

            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);

            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            if (albumChangedVisibility) {
                Option<RegionNode> regionO = geobase.getRegionById(info.regionId);
                if (regionO.isPresent() && regionO.get().getType().value() > RegionType.FEDERAL_SUBJECT.value()) {
                    removePhotosFromParentFederalSubjectAlbum(uid, regionO.get(), album.getId());
                }
            }

            return Option.of(new AlbumAddItemResultPojo(currentRevision, album, item));
        });

        addItemResultO.ifPresent(addItemResult -> {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, addItemResult.currentRevision);
            xivaPushGenerator.sendAlbumItemAppendPush(uid, addItemResult.album, addItemResult.item, true);
        });
    }

    public void startAlbumCreationProcess(DjfsUid uid) {
        if (isGeoAlbumsGeneratingInProgress(uid)) {
            return;
        }

        OnetimeTask task = diskInfoManager.getIndexerSyncGeoTimestamp(uid).isPresent() ?
                new GenerateGeoAlbumsTask(uid.asLong()) :
                new FetchCoordinatesFromIndexerTask(uid.asString(), true);

        try {
            bazingaTaskManager.schedule(task);
        } catch (Exception e) {  // do nothing here
            logger.error("Generate geo album task submit is failed", e);
        }
    }

    public boolean isGeoAlbumsGeneratingInProgress(DjfsUid uid) {
        return diskInfoManager.isGeoAlbumsGeneratingInProgress(uid);
    }

    private void removePhotosFromParentFederalSubjectAlbum(DjfsUid uid, RegionNode childRegion, ObjectId childAlbumId) {
        Assume.assumeTrue(childRegion.getType().value() > RegionType.FEDERAL_SUBJECT.value());

        Option<RegionNode> parentRegionO = geobase.getParentByType(Option.of(childRegion), RegionType.FEDERAL_SUBJECT);
        if (!parentRegionO.isPresent()) {
            return;
        }
        int parentGeoId = parentRegionO.get().getId();

        Option<Album> parentAlbumO = albumDao.findGeoAlbum(uid, parentGeoId);
        if (!parentAlbumO.isPresent()) {
            return;
        }

        ListF<AlbumItem> childAlbumItems = albumItemDao.getAllAlbumItems(uid, Cf.list(childAlbumId),
                visibilityThreshold);

        for (AlbumItem item : childAlbumItems) {
            DjfsResourceId itemResourceId = DjfsResourceId.cons(uid, item.getObjectId());
            removeFromGeoAlbum(itemResourceId, parentGeoId);
        }
    }

    private boolean checkChildAlbumVisibilityForFederalSubjectAlbum(DjfsUid uid, GeoInfo info,
                                                                    Option<Integer> childGeoId)
    {
        if (info.regionType == RegionType.FEDERAL_SUBJECT && childGeoId.isPresent()) {
            // this is workaround only for simultaneous updates
            // see test GeoAlbumGenerationTest.testSimultaneousPhotoRestoreToSeveralAlbums
            Option<Album> childAlbum = albumDao.findGeoAlbum(uid, childGeoId.get());
            return childAlbum.isPresent() && !childAlbum.get().isHidden();
        }
        return false;
    }

    public boolean updateCover(FileDjfsResource resource, GeoInfo info) {
        if (!resource.getAesthetics().isPresent()) {
            logger.info("No aesthetics for file is found: " + resource.getPath().getFullPath());
            return false;
        }
        Double resourceAesthetics = resource.getAesthetics().get();
        DjfsUid uid = resource.getUid();

        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            Option<ExtendedAlbum> albumO = albumDao.findExtendedGeoAlbum(uid, info.regionId);
            if (!albumO.isPresent()) {
                return Tuple2.tuple(false, 0L);
            }

            ExtendedAlbum extendedAlbum = albumO.get();
            ListF<AlbumItem> albumItems = albumItemDao
                    .findObjectInAlbum(uid, extendedAlbum.album.getId(), resource.getFileId().get().getValue());
            if (albumItems.isEmpty()) {
                logger.info("Item is not found in album");
                return Tuple2.tuple(false, 0L);
            }
            AlbumItem albumItem = albumItems.first();

            if (!shouldUpdateCover(resourceAesthetics, extendedAlbum)) {
                return Tuple2.tuple(false, 0L);
            }

            albumDao.setCover(extendedAlbum.album, albumItem.getId());
            ListF<AlbumDeltaFieldChange> albumUpdateChanges =
                    AlbumUtils.generateSetCoverChangeFields(albumItem.getId(), resource.getResourceId().get());

            currentRevision += 1;
            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(extendedAlbum.album, albumUpdateChanges,
                    currentRevision);
            albumDao.updateAlbumRevision(extendedAlbum.album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);

            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            ExtendedAlbum extendedAlbum = albumDao.findExtendedGeoAlbum(uid, info.regionId).get();;
            xivaPushGenerator.sendAlbumCoverChangedPush(uid, extendedAlbum.album);
            return true;
        }

        return false;
    }

    // temporary function to fix album titles according to user locale
    @Deprecated
    public void fixAlbumTitle(Album album, UserLocale userLocale) {
        if (album.getType() != AlbumType.GEO) {
            return;
        }
        DjfsUid uid = album.getUid();

        Tuple2<Boolean, Long> statusWithRevision = transactionUtils.executeInNewOrCurrentTransaction(uid, () -> {
            long currentRevision = getAndLockCurrentRevision(uid);

            String newTitle = makeGeoInfo(album.getGeoId().get().intValue(), userLocale.toLanguage()).regionName;

            albumDao.setTitle(album, newTitle);
            ListF<AlbumDeltaFieldChange> albumUpdateChanges = AlbumUtils.generateSetTitleChangeFields(newTitle);

            currentRevision += 1;
            AlbumDelta updateAlbumDelta = AlbumUtils.createUpdateAlbumDelta(album, albumUpdateChanges,
                    currentRevision);
            albumDao.updateAlbumRevision(album, currentRevision);
            albumDeltaDao.insert(updateAlbumDelta);

            albumDeltaDao.updateCurrentRevision(uid, currentRevision);

            return Tuple2.tuple(true, currentRevision);
        });

        if (statusWithRevision._1) {
            xivaPushGenerator.sendAlbumsDatabaseChangedPush(uid, statusWithRevision._2);
            xivaPushGenerator.sendAlbumTitleChangedPush(uid, album);
        }
    }

    private boolean shouldUpdateCover(Double resourceAesthetics, ExtendedAlbum extendedAlbum)
    {
        if (!extendedAlbum.coverResourceId.isPresent()) {
            return true;
        }

        DjfsResourceId currentCoverResourceId = extendedAlbum.coverResourceId.get();
        ListF<DjfsResource> resources = djfsResourceDao.find(currentCoverResourceId);

        if (resources.isEmpty() || !(resources.first() instanceof FileDjfsResource)) {
            return true;
        }

        FileDjfsResource currentCover = (FileDjfsResource) resources.first();
        return !currentCover.getAesthetics().isPresent() || currentCover.getAesthetics().get() < resourceAesthetics;
    }

    public boolean hasGeoAlbums(DjfsUid uid) {
        return albumDeltaDao.getCurrentRevisionWithoutLock(uid).isPresent();
    }

    public Language getLocale(DjfsUid uid) {
        Option<UserData> userData = userDao.find(uid);
        if (!userData.isPresent() || !userData.get().getLocale().isPresent()) {
            return defaultLocale;
        }
        return userData.get().getLocale().map(UserLocale::toLanguage).get();
    }

    public void postProcessFiles(ListF<DjfsResourceId> resourceIds) {
        SetF<DjfsUid> uids = resourceIds.map(DjfsResourceId::getUid).unique();
        SetF<DjfsUid> uidsWithAlbums = uids.filter(this::hasGeoAlbums);
        if (uidsWithAlbums.isEmpty()) {
            logger.info("No albums for users: " + String.join(", ", uids.map(DjfsUid::asString)));
            return;
        }

        resourceIds = resourceIds.filter(x -> uidsWithAlbums.containsTs(x.getUid()));
        MapF<DjfsUid, Language> uid2locale = uidsWithAlbums.toMap(u -> Tuple2.tuple(u, getLocale(u)));
        MapF<DjfsUid, ListF<DefaultCityStrategy>> uid2generationStrategies =
                uidsWithAlbums.toMap(u -> Tuple2.tuple(u, getGenerationStrategies(u)));
        ListF<DefaultCityStrategy> removeStrategies = getRemoveStrategies();

        for (DjfsResourceId resourceId : resourceIds) {
            ListF<DjfsResource> resources = djfsResourceDao.find(resourceId);
            if (resources.isEmpty()) {
                continue;
            }

            DjfsResource resource = resources.first();
            if (!(resource instanceof FileDjfsResource)) {
                logger.info("Resource is not file: " + resource.getPath().getFullPath());
                continue;
            }
            FileDjfsResource file = (FileDjfsResource) resource;

            if (!file.getCoordinates().isPresent()) {
                logger.info("File has no coordinates: " + resource.getPath().getFullPath());
                continue;
            }

            DjfsResourceArea area = file.getPath().getArea();

            if (area.equals(DjfsResourceArea.HIDDEN) || area.equals(DjfsResourceArea.TRASH)) {
                // remove from all albums
                CollectionF<Integer> geoIds = getAllResourceGeoIds(removeStrategies, file).values();
                if (geoIds.isEmpty()) {
                    logger.info("File has no geo ids: " + resource.getPath().getFullPath());
                    continue;
                }
                geoIds.forEach(i -> removeFromGeoAlbum(file, i));
            } else {
                // restore to albums with filtering
                DjfsUid uid = resource.getUid();

                ListF<GeoInfo> geoInfos = getResourceInfos(uid2generationStrategies.getTs(uid), file,
                        uid2locale.getTs(file.getUid()));
                if (geoInfos.isEmpty()) {
                    logger.info("File has no geo info: " + resource.getPath().getFullPath());
                    return;
                }

                Option<Integer> childRegionId = Option.empty();
                for (GeoInfo i : geoInfos) {
                    addPhotoToAlbum(file, i, childRegionId);
                    if (federalSubjectChildTypes.containsTs(i.regionType)) {
                        childRegionId = Option.of(i.regionId);
                    }
                }
            }
        }
    }

    private Album prepareAlbum(DjfsUid uid, long cityRegionId, String cityName, long revision, Instant modified) {
        Instant now = Instant.now();
        return Album.builder()
                .id(new ObjectId())

                .uid(uid)
                .type(AlbumType.GEO)
                .title(cityName)
                .description(Option.empty())
                .geoId(Option.of(cityRegionId))
                .revision(Option.of(revision))
                .hidden(true)

                .dateCreated(Option.of(now))
                .dateModified(Option.of(modified))

                .coverId(Option.empty())
                .coverOffsetY(Option.empty())

                .flags(Option.empty())
                .layout(Option.empty())

                .isBlocked(false)
                .blockReason(Option.empty())
                .fotkiAlbumId(Option.empty())

                .isPublic(false)
                .publicKey(Option.empty())
                .publicUrl(Option.empty())
                .shortUrl(Option.empty())
                .socialCoverStid(Option.empty())
                .isDescSorting(Option.empty())
                .albumItemsSorting(Option.empty())
                .build();
    }

    private AlbumItem prepareAlbumItem(Album album, DjfsFileId fileId, Long etime) {
        return AlbumItem.builder()
                .id(new ObjectId())
                .uid(album.getUid())
                .albumId(album.getId())
                .description(Option.empty())
                .groupId(Option.empty())
                .objectId(fileId.getValue())
                .objectType(AlbumItemType.RESOURCE)
                .orderIndex(Option.of(-(double)(etime)))
                .faceInfo(Option.empty())
                .dateCreated(Option.of(Instant.now()))
                .build();
    }

    @Data
    private static class CityUnderSubject {

        private final Integer subjectGeoId;

        private final Integer cityGeoId;
    }

    @Override
    protected Option<Double> itemOrderIndex(FileDjfsResource file) {
        final double time = file
                .getEtime()
                .getOrElse(() -> {
                    logger.info("No etime for file: " + file);
                    return InstantUtils.toSecondsLong(file.getCreationTime());
                });
        return Option.of(-time);
    }
}
