package ru.yandex.chemodan.app.djfs.core.album;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.client.DataApiHttpClient;
import ru.yandex.chemodan.app.djfs.core.client.ProfileAddress;
import ru.yandex.chemodan.app.djfs.core.client.ProfileAddressType;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.inside.geobase.Geobase;
import ru.yandex.inside.geobase.RegionNode;
import ru.yandex.inside.geobase.RegionType;
import ru.yandex.misc.geo.Coordinates;

public class GeoStrategyProvider {

    private final Geobase geobase;
    private final DataApiHttpClient dataApiHttpClient;

    private ListF<RegionType> USER_REGION_TYPES = Cf.list(
            RegionType.CITY,
            RegionType.VILLAGE,
            RegionType.SETTLEMENT
    );

    public GeoStrategyProvider(Geobase geobase, DataApiHttpClient dataApiHttpClient) {
        this.geobase = geobase;
        this.dataApiHttpClient = dataApiHttpClient;
    }

    public ListF<DefaultCityStrategy> get(DjfsUid uid) {
        ListF<ProfileAddressType> detailedAddressTypes = Cf
                .list(ProfileAddressType.HOME, ProfileAddressType.WORK);

        ListF<Tuple2<Integer, RegionType>> regions = Cf.arrayList();

        for (ProfileAddressType type : detailedAddressTypes) {
            Option<ProfileAddress> addressO = dataApiHttpClient.fetchAddress(uid, type);
            if (!addressO.isPresent()) {
                continue;
            }
            ProfileAddress address = addressO.get();

            Coordinates coordinates = new Coordinates(address.latitude, address.longitude);
            Option<Integer> regionIdO = geobase.getRegionIdByCoordinates(coordinates);

            if (!regionIdO.isPresent()) {
                continue;
            }

            Option<RegionNode> regionO = geobase.getRegionById(regionIdO.get());
            if (!regionO.isPresent()) {
                continue;
            }

            for (RegionType regionType : USER_REGION_TYPES) {
                Option<RegionNode> userRegionO = getRegionNodeByType(regionO, regionType);
                if (userRegionO.isPresent()) {
                    regions.add(Tuple2.tuple(userRegionO.get().getId(), regionType));
                    break;
                }
            }
        }

        if (regions.size() == 2 && regions.first()._1.equals(regions.last()._1)) {
            regions = regions.take(1);
        }

        if (regions.isNotEmpty()) {
            return regions.map(t -> resolveStrategy(geobase, t._2, t._1));
        }

        return Cf.list(new DefaultCityStrategy(geobase));
    }

    public AllAlbumsStrategy allAlbumsStrategy() {
        return new AllAlbumsStrategy(geobase);
    }

    private DefaultCityStrategy resolveStrategy(Geobase geobase, RegionType type, int geoId) {
        switch (type) {
            case CITY: return new NativeCityStrategy(geobase, geoId);
            case VILLAGE: return new NativeVillageStrategy(geobase, geoId);
            case SETTLEMENT: return new NativeSettlementStrategy(geobase, geoId);
        }
        return new DefaultCityStrategy(geobase);
    }

    private Option<RegionNode> getRegionNodeByType(Option<RegionNode> nodeO, RegionType type) {
        if (!nodeO.isPresent()) {
            return Option.empty();
        }
        return nodeO.get().getType() == type ? nodeO : geobase.getParentByType(nodeO, type);
    }
}
